%%%
%%% This script specifies all anatomical and physiological data of
%%% different species
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

function species = GenericPBPKmodel_speciesDatabase 

fprintf('\n\n   (a) Loading species database')


%%% ========================================================================================================
%%% CONTENT:
%%%
%%% 1) Data fields, units, description of parameters, tissue indexing
%%% 2) Rat
%%% 3) Mouse
%%% 4) Human
%%%
%%% ========================================================================================================


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: DATA FIELDS, UNITS, PARAMETER DESCRIPTION, TISSUE INDEXING ++++++++++++++++++++++++++++++++++++++
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% ========================================================================================================
%%% == DATA FIELDS, UNITS, and PARAMETER DESCRIPTION
%%%

data.subtype.type       = [];    info.type       = 'species type: rat, mouse, human';
data.subtype.subtype    = [];    info.subtype    = 'subtype (animal) or age&sex (human)';
data.subtype.info       = [];    info.info       = 'information on parameters';
data.subtype.T          = [];    info.T          = 'tissue indexing';
data.subtype.top        = [];    info.top        = 'default organ topology';
data.subtype.id         = [];    info.id         = 'id number';
data.subtype.color      = [];    info.color      = 'default color for plotting';

% age, sex, (lean) body weight, BMI, BSA
data.subtype.I          = '------>> physiological & anatomical data <<-------'; info.I = data.subtype.I;
data.subtype.age        = [];    info.age        = 'age of subtype in weeks(animals); years(human)';
data.subtype.sex        = [];    info.sex        = 'sex of subtype: m(ale)/f(emale)/u(nisex)';
data.subtype.BH         = [];    info.BH         = 'body height in [m]';
data.subtype.BW         = [];    info.BW         = 'body weight in [kg]';
data.subtype.LBW        = [];    info.LBW        = 'lean body weight in [kg]';
data.subtype.BMI        = [];    info.BMI        = 'body mass index in [kg/m^2]';
data.subtype.BSA        = [];    info.BSA        = 'body surface area in [m^2]';
   
% organ weight/fractions, tissue volumes and blood fractions 
data.subtype.OW         = [];    info.OW         = 'tissue weights (NOT INCLUDING VASCULAR SPACE) in [kg]';
data.subtype.fOW        = [];    info.fOW        = 'tissue weights as fraction of BW';
data.subtype.dens       = [];    info.dens       = 'tissue density in [kg/L]';
data.subtype.fvB        = [];    info.fvB        = 'tissue vascular blood as fraction of total blood';

% tissue volumes, constituent fractions and ratios (for partition coefficients)
data.subtype.V          = [];
data.subtype.V.tis      = [];    info.V.tis      = 'tissue volumes (NOT INCLUDING VASCULAR SPACE) in [L]';
data.subtype.V.vas      = [];    info.V.vas      = 'perihperal vascular blood in [L]';
data.subtype.V.tot      = [];    info.V.tot      = 'total tissue volumes (INCLUDING VASCULAR SPACE) in [L]';
data.subtype.hct        = [];    info.hct        = 'hematocrit value as fraction of red blood cells-to-blood volume fraction';
data.subtype.fVtis      = [];
data.subtype.fVtis.int  = [];    info.fVtis.int  = 'interstitial volume as fraction of tissue volume';
data.subtype.fVtis.cel  = [];    info.fVtis.cel  = 'intra-cellular volume as fraction of tissue volume';
data.subtype.fVtis.wex  = [];    info.fVtis.wex  = 'extra-cellular water volume as fraction of tissue volume';
data.subtype.fVtis.wic  = [];    info.fVtis.wic  = 'intra-cellular water volume as fraction of tissue volume';
data.subtype.fVtis.nlt  = [];    info.fVtis.nlt  = 'neutral lipid volume as fraction of tissue volume';
data.subtype.fVtis.npt  = [];    info.fVtis.npt  = 'neutral phospholipd volume as fraction of tissue volume';
data.subtype.fVtis.APmt = [];    info.fVtis.APmt = 'intra-cellular acidic phospholipid volume as fraction of tissue volume';
data.subtype.r          = [];    
data.subtype.r.Atp      = [];    info.r.Atp      = 'ratio of Albumin tissue-to-plasma concentrations';
data.subtype.r.Ltp      = [];    info.r.Ltp      = 'ratio of Lipoprotein tissue-to-plasma concentrations';

% cardiac output and blood flows/fractions
data.subtype.co         = [];    info.co         = 'cardiac output in [L/min]';
data.subtype.Q.blo      = [];    info.Q.blo      = 'peripheral blood flows in [L/min]';
data.subtype.fQco       = [];    info.fQco       = 'peripheral blood flows as fraction of cardiac output';


%%% ========================================================================================================
%%% Indexing of organs, tissues and other spaces
%%%
lun =  1; T.name{lun} = 'lun'; T.lun = lun; T.style.lun = 'g--';
art =  2; T.name{art} = 'art'; T.art = art; T.style.art = 'g-';
bra =  3; T.name{bra} = 'bra'; T.bra = bra; T.style.bra = 'm:';
adi =  4; T.name{adi} = 'adi'; T.adi = adi; T.style.adi = 'r--';
hea =  5; T.name{hea} = 'hea'; T.hea = hea; T.style.hea = 'm--';
kid =  6; T.name{kid} = 'kid'; T.kid = kid; T.style.kid = 'm-';
mus =  7; T.name{mus} = 'mus'; T.mus = mus; T.style.mus = 'b--';
bon =  8; T.name{bon} = 'bon'; T.bon = bon; T.style.bon = 'b-';
ski =  9; T.name{ski} = 'ski'; T.ski = ski; T.style.ski = 'r-';
gut = 10; T.name{gut} = 'gut'; T.gut = gut; T.style.gut = 'k--';
spl = 11; T.name{spl} = 'spl'; T.spl = spl; T.style.spl = 'k:';
liv = 12; T.name{liv} = 'liv'; T.liv = liv; T.style.liv = 'k-';
ven = 13; T.name{ven} = 'ven'; T.ven = ven; T.style.ven = 'g-';
rob = 14; T.name{rob} = 'rob'; T.rob = rob; T.style.rob = 'y-';
blo = 15; T.name{blo} = 'blo'; T.blo = blo; T.style.blo = 'g-';
pla = 16; T.name{pla} = 'pla'; T.pla = pla; T.style.pla = 'g-';
ery = 17; T.name{ery} = 'ery'; T.ery = ery; T.style.ery = 'g:';

%%% tissue database (DB)
T.tissueDB = [lun art bra adi hea kid mus bon ski gut spl liv ven rob blo pla ery];
T.initialize.tissueDB.NaN = NaN*ones(1,max(T.tissueDB)); 


%%% Rest of body weight as well as lun, art & ven blood flows 
%%% are determined based on the following reference model
%%%
refModel.T = T;
refModel.T.allTis      = [lun art bra adi hea kid mus bon ski gut spl liv ven];
refModel.T.allTisExBlo = [lun     bra adi hea kid mus bon ski gut spl liv    ];
refModel.T.artInflow   = [        bra adi hea kid mus bon ski gut spl        ];
refModel.T.intoLiv     = [                                    gut spl        ];
refModel.T.intoVen     = [        bra adi hea kid mus bon ski         liv    ];
refModel.T.elimTis     = [                                            liv    ];

%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: DATA FIELDS, UNITS, PARAMETER DESCRIPTION, TISSUE INDEXING ++++++++++++++++++++++++++++++++++++++++
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++++ 
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% ========================================================================================================
%%% Data: Mostly reported for F344 rats
%%% 
%%% Source: Brown et al, Tox Ind Health 1997 (Table 3)
%%%
%%% Note: Male rats of 250g are only 10 weeks of age and in rapid growth phase.
%%% Growth is much slower between age 20-55 weeks (ca. 380-470g) and reaches 
%%% a plateau of 475g at age 56-91 weeks.  

% rat250 = 1; rat475 = 2;

rat.type    = 'rat';
rat.subtype = {'rat250','rat475'};


%%% ========================================================================================================
%%% Data  : Body weight (BW)
%%%
%%% Units : BW in kg, 
%%% Source: Brown et al, Tox Ind Health 1997 (relation between weight and age, see Table 3)
%%%
%%% Strain: most data were reported for the F344 strain

data.rat.sex =   {'m','m'};
data.rat.age =   [ 10 70 ]; % in weeks; 70w = between 56-91w
data.rat.BW  =   [ 0.250 0.475]; % 

%%% Assign values to all age/sex classes
for a = 1:length(rat.subtype)   
    
    subtype = rat.subtype{a};
    rat.(subtype)         = struct(data.subtype);
    rat.(subtype).type    = 'rat';
    rat.(subtype).subtype = subtype;
    rat.(subtype).info    = info;
    rat.(subtype).T       = T;
    rat.(subtype).age     = data.rat.age(a);  
    rat.(subtype).sex     = data.rat.sex{a};
    rat.(subtype).BW      = data.rat.BW(a);   
    rat.(subtype).BH      = NaN;              
    rat.(subtype).LBW     = NaN;              
    rat.(subtype).BMI     = NaN;              
    rat.(subtype).BSA     = NaN;              
          
end;


%%% ========================================================================================================
%%% Data  : Density of tissue
%%%
%%% Unit  : kg/l (= g/cm^3)
%%% Source: Brown et al, Tox Ind Health 1997 and ICRP report 1975 
%%% 
data.human.dens      = ones(size(T.initialize.tissueDB.NaN)); % all organs except for adi and bon (see Brown et al, Tox Ind Health 1997, Table 19)
data.human.dens(adi) = 0.916;                  % ICRP report 1975, p.44
data.human.dens(bon) = 1.3;                    % ICRP report 2002, Table 2.20 (whole skeleton, adults)
%%%
%%% Due to LACK OF DATA
%%%
data.rat.dens = data.human.dens;


%%% ========================================================================================================
%%% Data  : relative organ weight as fraction of total body weight
%%%
%%% Unit  : fraction (converted from percentage by dividing by 100]
%%% Source: Brown et al, Tox Ind Health 1997: Table 5 (most tissues), 
%%%         Table 13 (adi), top paragraph on p.425 (bone), Table 21 (blo)
%%%

%%%    age(weeks)    10w   70w
data.rat.fOW.adi = [10.6  16.0 ]/100; 
data.rat.fOW.bon = [ 7.3   7.3 ]/100; 
data.rat.fOW.bra = [ 0.57  0.57]/100; 
data.rat.fOW.gut = [ 2.7   2.7 ]/100; % sum of stomach, small and large intestine (2.7=0.46+1.40+0.84) 
data.rat.fOW.hea = [ 0.33  0.33]/100;
data.rat.fOW.kid = [ 0.73  0.73]/100;
data.rat.fOW.liv = [ 3.66  3.66]/100;
data.rat.fOW.lun = [ 0.50  0.50]/100;
data.rat.fOW.mus = [40.43 40.43]/100;
data.rat.fOW.ski = [19.03 19.03]/100;
data.rat.fOW.spl = [ 0.2   0.2 ]/100;
data.rat.fOW.blo = [ 7.4   7.4 ]/100; 

for a = 1:length(rat.subtype)   
    
    subtype = rat.subtype{a};
    rat.(subtype).fOW      = T.initialize.tissueDB.NaN;
    rat.(subtype).fOW(adi) = data.rat.fOW.adi(a);
    rat.(subtype).fOW(bon) = data.rat.fOW.bon(a);
    rat.(subtype).fOW(bra) = data.rat.fOW.bra(a);
    rat.(subtype).fOW(gut) = data.rat.fOW.gut(a);
    rat.(subtype).fOW(hea) = data.rat.fOW.hea(a);
    rat.(subtype).fOW(kid) = data.rat.fOW.kid(a);
    rat.(subtype).fOW(liv) = data.rat.fOW.liv(a);
    rat.(subtype).fOW(lun) = data.rat.fOW.lun(a);
    rat.(subtype).fOW(mus) = data.rat.fOW.mus(a);
    rat.(subtype).fOW(ski) = data.rat.fOW.ski(a);
    rat.(subtype).fOW(spl) = data.rat.fOW.spl(a);    
    rat.(subtype).fOW(blo) = data.rat.fOW.blo(a);
    
    
    %%% DUE TO LACK OF DATA:
    %%%
    %%% Assume that 'ven' and 'art' are 2/3 and 1/3 of 'blo' and
    %%% set the hematocrit value to 0.45
    %%%
    rV_ven_blo = 2/3; 
    rat.(subtype).fOW(ven) =    rV_ven_blo *rat.(subtype).fOW(blo);
    rat.(subtype).fOW(art) = (1-rV_ven_blo)*rat.(subtype).fOW(blo);
    
    rat.(subtype).hct        = 0.45;

    rat.(subtype).fOW(rob) = 1-sum(rat.(subtype).fOW(refModel.T.allTis));
    rat.(subtype).fOW(ery) = rat.(subtype).hct * rat.(subtype).fOW(blo);
    rat.(subtype).fOW(pla) = (1-rat.(subtype).hct) * rat.(subtype).fOW(blo);
       
    rat.(subtype).OW       = rat.(subtype).fOW * rat.(subtype).BW;
    rat.(subtype).LBW      = rat.(subtype).BW  - rat.(subtype).OW(adi); 
    
    rat.(subtype).dens     = data.rat.dens;

    rat.(subtype).V.tis    = rat.(subtype).OW ./ rat.(subtype).dens;
    
    %%% V.vas and V.tot are defined below
end;


%%% ========================================================================================================
%%% Data  : Fraction of vascular space (fVtis.vas) and residual blood (fVtis.res) in rat (based on
%%%         tissue weight INCLUDING vascular blood!)
%%%
%%% Unit  : fraction
%%% Source: Kawai etal, J Pharmacokinet Biopharm, Vol 22, 1994 (Table B-I),
%%%         based on measurements in nonbled (fVtis.vas) and exanguinated (fVtis.res) rats
%%%
data.rat.fVtis.vas = T.initialize.tissueDB.NaN; % vascular fraction
data.rat.fVtis.vas(adi) = 0.010; 
data.rat.fVtis.vas(bon) = 0.041; 
data.rat.fVtis.vas(bra) = 0.037; 
data.rat.fVtis.vas(gut) = 0.024; 
data.rat.fVtis.vas(hea) = 0.262;
data.rat.fVtis.vas(kid) = 0.105;
data.rat.fVtis.vas(liv) = 0.115;
data.rat.fVtis.vas(lun) = 0.262;
data.rat.fVtis.vas(mus) = 0.026;
data.rat.fVtis.vas(ski) = 0.019;
data.rat.fVtis.vas(spl) = 0.282;

data.rat.fVtis.res = T.initialize.tissueDB.NaN; % residual blood volume fraction. For interpreation of Brown et al 1997,
data.rat.fVtis.res(adi) = 0.005; % Table 30, see 2nd paragraph on p.457 under BLOOD VOLUME DATA
data.rat.fVtis.res(bon) = 0.019; % Brown etal 1997(Table 30) = 0.04 
data.rat.fVtis.res(bra) = 0.014; % Brown etal 1997(Table 30) = [0.02-0.04]
data.rat.fVtis.res(gut) = 0.010; 
data.rat.fVtis.res(hea) = 0.061; % Brown etal 1997(Table 30) = 0.26
data.rat.fVtis.res(kid) = 0.046; % Brown etal 1997(Table 30) = [0.11-0.27], mean 0.16
data.rat.fVtis.res(liv) = 0.057; % Brown etal 1997(Table 30) = [0.12-0.27], mean 0.21
data.rat.fVtis.res(lun) = 0.175; % Brown etal 1997(Table 30) = [0.26-0.52], mean 0.36
data.rat.fVtis.res(mus) = 0.004; % Brown etal 1997(Table 30) = [0.01-0.04], mean 0.04
data.rat.fVtis.res(ski) = 0.002; % Brown etal 1997(Table 30) = 0.02
data.rat.fVtis.res(spl) = 0.321; % Brown etal 1997(Table 30) = [0.17-0.28], mean 0.22


%%% ========================================================================================================
%%% Data  : Fraction of interstitial and intra-cellular space (fVtis.int, fVtis.cel) in rat
%%%         (based on tissue weight INCLUDING vascular blood!)
%%%
%%% Unit  : fraction of tisse+vascular space (original) converted to fraction of tissue space
%%% Source: Kawai etal, J Pharmacokinet Biopharm, Vol 22, 1994 (Table B-I), identical to 
%%%         Graham etal, J Pharm Pharmacol (2011) - except for adipose (0.140 vs. 0.135)
%%%
data.rat.fVtis.int = T.initialize.tissueDB.NaN;
data.rat.fVtis.int(adi) = 0.135;
data.rat.fVtis.int(bon) = 0.100; 
data.rat.fVtis.int(bra) = 0.004; 
data.rat.fVtis.int(gut) = 0.094; 
data.rat.fVtis.int(hea) = 0.100;
data.rat.fVtis.int(kid) = 0.200;
data.rat.fVtis.int(liv) = 0.163;
data.rat.fVtis.int(lun) = 0.188;
data.rat.fVtis.int(mus) = 0.120;
data.rat.fVtis.int(ski) = 0.302;
data.rat.fVtis.int(spl) = 0.150;
%%%
data.rat.fVtis.cel = 1-data.rat.fVtis.vas-data.rat.fVtis.int;
%%%
%%% Determining fraction of interstitial and intra-cellular space with
%%% respect to tissue weight NOT INCLUDING vascular blood so that
%%% fVtis.int+fVtis.cel = 1, hence
%%% Scaling: fraction of tissue+vascular space --> fraction of tissue space
%%%
data.rat.fVtis.int = data.rat.fVtis.int./(1-data.rat.fVtis.vas);
data.rat.fVtis.cel = data.rat.fVtis.cel./(1-data.rat.fVtis.vas);


%%% ========================================================================================================
%%% Data  : Regional blood volumes as percentage of total blood volume
%%%
%%% Unit  : percent (%)
%%% Source: derived from Kawai etal, J Pharmacokinet Biopharm, Vol 22, 1994 (Table B-I),
%%%         and Brown et al, Tox Ind Health 1997. 
%%%

allTis = refModel.T.allTis; m = refModel.T.allTisExBlo;  b = [blo ven art];
for a = 1:length(rat.subtype)   
    
    subtype = rat.subtype{a};
    rat.(subtype).fvB   = T.initialize.tissueDB.NaN;

    %%% determine peripheral vascular blood vasB and the fraction of blood 
    %%% not associated with any tissue
    vasB      = T.initialize.tissueDB.NaN;
    vasB(m)   = data.rat.fVtis.vas(m).*rat.(subtype).OW(m);
    vasB(blo) = rat.(subtype).OW(blo)-sum(vasB(m));

    %%% maintain volume ratio between artery and vein compartment
    ratio = rat.(subtype).OW(ven)/sum(rat.(subtype).OW([art ven]));
    vasB(ven) =    ratio *vasB(blo);
    vasB(art) = (1-ratio)*vasB(blo);
    
    rat.(subtype).fvB         = T.initialize.tissueDB.NaN;
    rat.(subtype).fvB(allTis) = vasB(allTis)/rat.(subtype).OW(blo);
    rat.(subtype).fvB(blo)    = rat.(subtype).fvB(art)+rat.(subtype).fvB(ven);
    
    %%% define V.vas and V.tot
    rat.(subtype).V.vas   = rat.(subtype).fvB*rat.(subtype).V.tis(blo);
    
    rat.(subtype).V.tot   = T.initialize.tissueDB.NaN;
    rat.(subtype).V.tot(m)= rat.(subtype).V.vas(m) + rat.(subtype).V.tis(m);
    rat.(subtype).V.tot(b)= rat.(subtype).V.vas(b);
    
    
end;



%%% ========================================================================================================
%%% Data  : regional blood flows
%%%
%%% Unit  : fraction of co (in L/min)
%%% Source: Brown et al, Tox Ind Health 1997 (Table 25)
%%%
for a = 1:length(rat.subtype)   
    
    subtype = rat.subtype{a};
    rat.(subtype).co = 0.235*(rat.(subtype).BW^0.75); % Brown et al, Tox Ind Health 1997, p.441
end;

data.rat.fQco = T.initialize.tissueDB.NaN;
data.rat.fQco(adi) =  7.0/100; 
data.rat.fQco(bon) = 12.2/100; 
data.rat.fQco(bra) =  2.0/100; 
data.rat.fQco(gut) = 13.1/100; % Ref: ?? 
data.rat.fQco(hea) =  4.9/100;
data.rat.fQco(kid) = 14.1/100;
data.rat.fQco(liv) = 17.4/100;
data.rat.fQco(mus) = 27.8/100;
data.rat.fQco(ski) =  5.8/100;
data.rat.fQco(spl) =  2.0/100; % Poulin & Theil, J Pharm Sci 2002

data.rat.fQco(rob) = 1 - sum(data.rat.fQco(refModel.T.intoVen));
data.rat.fQco(lun) = 1 - data.rat.fQco(rob); % ensures closed blood flow system
data.rat.fQco(art) = data.rat.fQco(lun);
data.rat.fQco(ven) = data.rat.fQco(lun);


for a = 1:length(rat.subtype)   
    
    subtype = rat.subtype{a};
    rat.(subtype).fQco = data.rat.fQco;
    rat.(subtype).Q.blo = rat.(subtype).fQco * rat.(subtype).co;
end;


%%% ========================================================================================================
%%% Data  : Total tissue water (Vwt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: Rodgers, Leahy, and Rowland, J Pharm Sci (2005), identical to Graham etal, J Pharm Pharmacol (2011)
%%% Note  : In Rodgers and Rowland, J Pharm Sci (2006), Table 1, total tissue water is reported, including 
%%%         correction for residual blood. The values below are not corrected for residual blood and hence 
%%%         fVtis.wex+fVtis.wic is larger than the total tissue water reported in Rodgers and Rowland (2006)
%%%
data.rat.fVtis.wex = T.initialize.tissueDB.NaN;  %%% tissue water (extra-cellular)
data.rat.fVtis.wex(adi) = 0.135;
data.rat.fVtis.wex(bon) = 0.100;
data.rat.fVtis.wex(bra) = 0.162;
data.rat.fVtis.wex(gut) = 0.282;
data.rat.fVtis.wex(hea) = 0.320;
data.rat.fVtis.wex(kid) = 0.273;
data.rat.fVtis.wex(liv) = 0.161;
data.rat.fVtis.wex(lun) = 0.336;
data.rat.fVtis.wex(mus) = 0.118;  
data.rat.fVtis.wex(ski) = 0.382;
data.rat.fVtis.wex(spl) = 0.207;
data.rat.fVtis.wex(pla) = NaN;
data.rat.fVtis.wex(ery) = 0; 
%%%
data.rat.fVtis.wic = T.initialize.tissueDB.NaN;  %%% tissue water (intra-cellular)
data.rat.fVtis.wic(adi) = 0.017;
data.rat.fVtis.wic(bon) = 0.346;
data.rat.fVtis.wic(bra) = 0.620;
data.rat.fVtis.wic(gut) = 0.475;
data.rat.fVtis.wic(hea) = 0.456;
data.rat.fVtis.wic(kid) = 0.483;
data.rat.fVtis.wic(liv) = 0.573;
data.rat.fVtis.wic(lun) = 0.446;
data.rat.fVtis.wic(mus) = 0.630;  
data.rat.fVtis.wic(ski) = 0.291;
data.rat.fVtis.wic(spl) = 0.579;
data.rat.fVtis.wic(pla) = NaN;
data.rat.fVtis.wic(ery) = 0.603; 


%%% ========================================================================================================
%%% Data  : Neutral lipids (fVtis.nlt) and neutal phospholipids (fVtis.npt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: Rodgers and Rowland, J Pharm Sci (2006), identical to Graham etal, J Pharm Pharmacol (2011)
%%%
data.rat.fVtis.nlt = T.initialize.tissueDB.NaN;  %%% neutral lipids
data.rat.fVtis.nlt(adi) = 0.8530; % note: in Rodgers and Rowland (2006) incorrectly reported under neutral phospholipids
data.rat.fVtis.nlt(bon) = 0.0174; 
data.rat.fVtis.nlt(bra) = 0.0391;
data.rat.fVtis.nlt(gut) = 0.0375;
data.rat.fVtis.nlt(hea) = 0.0135;
data.rat.fVtis.nlt(kid) = 0.0121;
data.rat.fVtis.nlt(liv) = 0.0135;
data.rat.fVtis.nlt(lun) = 0.0215;
data.rat.fVtis.nlt(mus) = 0.0100;  
data.rat.fVtis.nlt(ski) = 0.0603;
data.rat.fVtis.nlt(spl) = 0.0071;
data.rat.fVtis.nlt(pla) = 0.0023; % p. 1241, paragarph "Tissue Specific Input Parameters"
data.rat.fVtis.nlt(ery) = 0.0017; % Rodgers, Leahy, Rowland (2005), Table 1 
%%%
data.rat.fVtis.npt = T.initialize.tissueDB.NaN;  %%% neutral phospholipids 
data.rat.fVtis.npt(adi) = 0.0016; % note: in Rodgers and Rowland (2006) incorrectly reported under neutral lipids
data.rat.fVtis.npt(bon) = 0.0016; 
data.rat.fVtis.npt(bra) = 0.0015;
data.rat.fVtis.npt(gut) = 0.0124;
data.rat.fVtis.npt(hea) = 0.0106;
data.rat.fVtis.npt(kid) = 0.0240;
data.rat.fVtis.npt(liv) = 0.0238;
data.rat.fVtis.npt(lun) = 0.0123;
data.rat.fVtis.npt(mus) = 0.0072;  
data.rat.fVtis.npt(ski) = 0.0044;
data.rat.fVtis.npt(spl) = 0.0107; 
data.rat.fVtis.npt(pla) = 0.0013; % p. 1241, paragarph "Tissue Specific Input Parameters"
data.rat.fVtis.npt(ery) = 0.0029; % Rodgers, Leahy, Rowland (2005), Table 1


%%% ========================================================================================================
%%% Data  : Intra-cellular acidic phospholipids (APmt) in rat
%%%
%%% Unit  : mg/g tissue (original) scalled to fraction
%%% Source: Rodgers, Leahy, and Rowland, J Pharm Sci (2005), identical to Graham etal, J Pharm Pharmacol (2011)
%%%
data.rat.fVtis.APmt = T.initialize.tissueDB.NaN;
data.rat.fVtis.APmt(adi) = 0.40;
data.rat.fVtis.APmt(bon) = 0.67;
data.rat.fVtis.APmt(bra) = 0.40;
data.rat.fVtis.APmt(gut) = 2.41;
data.rat.fVtis.APmt(hea) = 2.25;
data.rat.fVtis.APmt(kid) = 5.03;
data.rat.fVtis.APmt(liv) = 4.56;
data.rat.fVtis.APmt(lun) = 3.91;
data.rat.fVtis.APmt(mus) = 1.53;
data.rat.fVtis.APmt(ski) = 1.32;
data.rat.fVtis.APmt(spl) = 3.18;
data.rat.fVtis.APmt(pla) = NaN;
data.rat.fVtis.APmt(ery) = 0.5;
data.rat.fVtis.APmt = data.rat.fVtis.APmt/1000;  % scaling from: mg/g -->  fraction


%%% ========================================================================================================
%%% Data  : Albumin (Atp) in rat
%%%
%%% Unit  : Tissue-to-plasma ratio
%%% Source: Rodgers, and Rowland, J Pharm Sci 95 (2006), identical to Graham etal, J Pharm Pharmacol (2011)
%%% 
data.rat.r.Atp  = T.initialize.tissueDB.NaN;
data.rat.r.Atp(adi) = 0.049;
data.rat.r.Atp(bon) = 0.100;
data.rat.r.Atp(bra) = 0.048;  
data.rat.r.Atp(gut) = 0.158;
data.rat.r.Atp(hea) = 0.157;
data.rat.r.Atp(kid) = 0.130;
data.rat.r.Atp(liv) = 0.086;
data.rat.r.Atp(lun) = 0.212;
data.rat.r.Atp(mus) = 0.064;
data.rat.r.Atp(ski) = 0.277;
data.rat.r.Atp(spl) = 0.097;


%%% ========================================================================================================
%%% Data  : Lipoprotein (Ltp) in rat
%%%
%%% Unit  : Tissue-to-plasma ratio
%%% Source: Rodgers, and Rowland, J Pharm Sci 95 (2006), identical to Graham etal, J Pharm Pharmacol (2011)
%%% 
data.rat.r.Ltp = T.initialize.tissueDB.NaN;
data.rat.r.Ltp(adi) = 0.068;
data.rat.r.Ltp(bon) = 0.050;
data.rat.r.Ltp(bra) = 0.041;            
data.rat.r.Ltp(gut) = 0.141;
data.rat.r.Ltp(hea) = 0.160;
data.rat.r.Ltp(kid) = 0.137;
data.rat.r.Ltp(liv) = 0.161;
data.rat.r.Ltp(lun) = 0.168;
data.rat.r.Ltp(mus) = 0.059;
data.rat.r.Ltp(ski) = 0.096;
data.rat.r.Ltp(spl) = 0.207;

%%% ========================================================================================================
%%% Model : Assignment of fractional volumes and ratio to all age classes
%%%
for a = 1:length(rat.subtype)
    
    subtype = rat.subtype{a};
    rat.(subtype).fVtis = data.rat.fVtis; 
    rat.(subtype).r     = data.rat.r; 
    
end;


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++++++ 
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++++ 
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% ========================================================================================================
%%% Data: Mostly reported for B6C3F1 mice
%%% 
%%% Source: Brown et al, Tox Ind Health 1997 (Table 1)
%%%
%%% Note: Male mice of 25g are only 9 weeks of age and in rapid growth phase.
%%% Growth is much slower between age 20-67 weeks (ca. 31.5-40 g) and reaches 
%%% a plateau of 40g at age 68-91 weeks.  

% mouse25 = 1; mouse40 = 2; mouseBax = 3; mouseFaH = 4;

mouse.type    = 'mouse';
mouse.subtype = {'mouse25','mouse40','mouseBax','mouseFaH'};


%%% ========================================================================================================
%%% Data  : Body weight (BW)
%%%
%%% Units : BW in kg, 
%%% Source: Brown et al, Tox Ind Health 1997 (Table 1)
%%% Strain: most data were reported for the B6C3F1 strain

data.mouse.sex =   {'NaN','NaN', 'NaN', 'NaN'};
data.mouse.age =   [ 9     70     NaN    NaN]; % in weeks
data.mouse.BW  =   [0.025 0.040 0.022  0.025]; % 

%%% Assign values to all age/sex classes
for a = 1:length(mouse.subtype)   
    
    subtype = mouse.subtype{a};
    mouse.(subtype)         = struct(data.subtype);
    mouse.(subtype).type    = 'mouse';
    mouse.(subtype).subtype = subtype;
    mouse.(subtype).info    = info;
    mouse.(subtype).T       = T;
    mouse.(subtype).age     = data.mouse.age(a);  
    mouse.(subtype).sex     = data.mouse.sex{a};
    mouse.(subtype).BW      = data.mouse.BW(a);   
    mouse.(subtype).BH      = NaN;              
    mouse.(subtype).LBW     = NaN;              
    mouse.(subtype).BMI     = NaN;              
    mouse.(subtype).BSA     = NaN;              
          
end;


%%% ========================================================================================================
%%% Data  : Density of tissue
%%%
%%% Unit  : kg/L (= g/mL = g/cm^3)
%%% Source: Brown et al, Tox Ind Health 1997 and ICRP report 1975 
%%% 
data.human.dens      = ones(size(T.initialize.tissueDB.NaN)); % all organs except for adi and bon (see Brown et al, Tox Ind Health 1997, Table 19)
data.human.dens(adi) = 0.916;                  % ICRP report 1975, p.44
data.human.dens(bon) = 1.3;                    % ICRP report 2002, Table 2.20 (whole skeleton, adults)
%%%
%%% Due to LACK OF DATA
%%%
data.mouse.dens = data.human.dens;


%%% ========================================================================================================
%%% Data  : relative organ weight as fraction of total body weight
%%%
%%% Unit  : fraction (converted from percentage by dividing by 100]
%%% Source: 1st and 2nd column:
%%%         Brown et al, Tox Ind Health 1997: Table 4 (most tissues), 
%%%         Table 10 (adi), Table 21 (adi,blo) 
%%%         3rd column:
%%%         Baxter et al, Cancer Research 54 (1994), Table 1 
%%%         (volumes were converted to OW with density = 1.0, except 
%%%         for bone = 1.5. Subsequently, OW were converted to fOW 
%%%         based on BW = 22g, as mentioned in the article (Material and Methods, p.1517). 
%%%         fOW_adi and fOW_bra were taken from Brown)
%%%         4th column: mouse from Fronton and Huisinga based on Baxter et al.

    % age(weeks)        
data.mouse.fOW.adi = [ 7   /100,  7   /100, 7/100   , 1.749/25]; % 4th column: volume 1.909 converted to weight 1.7486 using a density of 0.916 g/mL, Brown: highly strain and age dependent according to Brown, p.419 (top)
data.mouse.fOW.bon = [10.73/100, 10.73/100, 2.250/22, 2.535/25]; % 3rd column: volume (1.5) converted to weight based on dens = 1.5 as used by Baxter, 4th column: volume 1.950 converted to weight 2.5350 using a density of 1.3 g/mL, 
data.mouse.fOW.bra = [ 1.65/100,  1.65/100, 1.65/100, 1.65/100]; 
data.mouse.fOW.gut = [ 4.22/100,  4.22/100, 3.450/22, 3.871/25]; % 1st & 2nd column: sum of stomach, small and large intestine (4.22=0.60+2.53+1.09) 
data.mouse.fOW.hea = [ 0.50/100,  0.50/100, 0.133/22, 0.144/25];
data.mouse.fOW.kid = [ 1.67/100,  1.67/100, 0.298/22, 0.318/25];
data.mouse.fOW.liv = [ 5.49/100,  5.49/100, 0.951/22, 1.024/25];
data.mouse.fOW.lun = [ 0.73/100,  0.73/100, 0.191/22, 0.189/25];
data.mouse.fOW.mus = [38.40/100, 38.40/100, 7.924/22, 8.948/25];
data.mouse.fOW.ski = [16.53/100, 16.53/100, 2.940/22, 3.315/25];
data.mouse.fOW.spl = [ 0.35/100,  0.35/100, 0.100/22, 0.089/25]; 
data.mouse.fOW.blo = [ 4.9 /100,  4.9 /100, 1.407/22, 3.046/25]; % 3rd+4th column: plasma volume 0.774(3rd) and 1.675 (4th) converted to blood via V_blo = V_pla/(1-hct)

for a = 1:length(mouse.subtype)   
    
    subtype = mouse.subtype{a};
    mouse.(subtype).fOW      = T.initialize.tissueDB.NaN;
    mouse.(subtype).fOW(adi) = data.mouse.fOW.adi(a);
    mouse.(subtype).fOW(bon) = data.mouse.fOW.bon(a);
    mouse.(subtype).fOW(bra) = data.mouse.fOW.bra(a);
    mouse.(subtype).fOW(gut) = data.mouse.fOW.gut(a);
    mouse.(subtype).fOW(hea) = data.mouse.fOW.hea(a);
    mouse.(subtype).fOW(kid) = data.mouse.fOW.kid(a);
    mouse.(subtype).fOW(liv) = data.mouse.fOW.liv(a);
    mouse.(subtype).fOW(lun) = data.mouse.fOW.lun(a);
    mouse.(subtype).fOW(mus) = data.mouse.fOW.mus(a);
    mouse.(subtype).fOW(ski) = data.mouse.fOW.ski(a);
    mouse.(subtype).fOW(spl) = data.mouse.fOW.spl(a);    
    mouse.(subtype).fOW(blo) = data.mouse.fOW.blo(a);
    
    
    %%% DUE TO LACK OF DATA:
    %%%
    %%% Assume that 'ven' and 'art' are 2/3 and 1/3 of 'blo' and
    %%% set the hematocrit value to 0.45
    %%%
    mouse.(subtype).fOW(ven) = (2/3)*mouse.(subtype).fOW(blo);
    mouse.(subtype).fOW(art) = (1/3)*mouse.(subtype).fOW(blo);
    
    mouse.(subtype).hct        = 0.45;

    mouse.(subtype).fOW(rob) = 1-sum(mouse.(subtype).fOW(refModel.T.allTis));
    mouse.(subtype).fOW(ery) = mouse.(subtype).hct * mouse.(subtype).fOW(blo);
    mouse.(subtype).fOW(pla) = (1-mouse.(subtype).hct) * mouse.(subtype).fOW(blo);
       
    mouse.(subtype).OW       = mouse.(subtype).fOW * mouse.(subtype).BW;
    mouse.(subtype).LBW      = mouse.(subtype).BW  - mouse.(subtype).OW(adi); 
    
    mouse.(subtype).dens     = data.mouse.dens;

    mouse.(subtype).V.tis    = mouse.(subtype).OW ./ mouse.(subtype).dens;
end;

%%% ========================================================================================================
%%% Data  : Fraction of vascular space (fVtis.vas) and residual blood (fVtis.res) in mouse 
%%%
%%% Unit  : fraction of tissue space
%%% Source: vascular fractions, Baxter et al (1994), Table 1 (these are assumed=educatedly guessed values!, 
%%%         see footnote 'd' of Table 1), Kawai et al (1994) for adi and bra
%%%         residual fractions, Garg, PhD thesis (2007), Table III, p.105,
%%%         reported as ratio of the residual blood volume to the tissue volume in mL/100g tissue
%%%
data.mouse.fVtis.vas = T.initialize.tissueDB.NaN;
data.mouse.fVtis.vas(adi) = 0.010; 
data.mouse.fVtis.vas(bon) = 0.080/1.500; % approx  5%
data.mouse.fVtis.vas(bra) = 0.037; 
data.mouse.fVtis.vas(gut) = 0.100/3.450; % approx  3%
data.mouse.fVtis.vas(hea) = 0.007/0.133; % approx  5%
data.mouse.fVtis.vas(kid) = 0.030/0.298; % approx 10%
data.mouse.fVtis.vas(liv) = 0.095/0.951; % approx 10%
data.mouse.fVtis.vas(lun) = 0.019/0.191; % approx 10%
data.mouse.fVtis.vas(mus) = 0.150/7.924; % approx  2%
data.mouse.fVtis.vas(ski) = 0.200/2.940; % approx  7%
data.mouse.fVtis.vas(spl) = 0.010/0.100; % approx 10%

data.mouse.fVtis.res = T.initialize.tissueDB.NaN; % residual blood fraction. For interpreation of Brown et al 1997,
data.mouse.fVtis.res(adi) =   NaN/100; % Table 30, see 2nd paragraph on p.457 under BLOOD VOLUME DATA
data.mouse.fVtis.res(bon) =   NaN/100; % Brown etal 1997(Table 30) = 0.11 
data.mouse.fVtis.res(bra) =   NaN/100; % Brown etal 1997(Table 30) = 0.03
data.mouse.fVtis.res(gut) =  1.27/100; 
data.mouse.fVtis.res(hea) =  4.81/100; 
data.mouse.fVtis.res(kid) =  6.23/100; % Brown etal 1997(Table 30) = [0.12-0.34], mean 0.24
data.mouse.fVtis.res(liv) =  5.27/100; % Brown etal 1997(Table 30) = [0.23-0.36], mean 0.31 
data.mouse.fVtis.res(lun) = 13.13/100; % Brown etal 1997(Table 30) = [0.40-0.62], mean 0.50
data.mouse.fVtis.res(mus) =  0.63/100; % Brown etal 1997(Table 30) = [0.03-0.05], mean 0.04
data.mouse.fVtis.res(ski) =  0.77/100; % Brown etal 1997(Table 30) = 0.03
data.mouse.fVtis.res(spl) = 21.51/100; % Brown etal 1997(Table 30) = [0.17-0.19], mean 0.17


%%% ========================================================================================================
%%% Data  : Fraction of interstitial and intra-cellular space (fVtis.int, fVtis.cel) in mouse
%%%
%%% Unit  : fraction of joint tisse+vascular space (original) converted to fraction of tissue space
%%% Source: Baxter et al (1994), Table 1 and Kawai et al (1994) for adi and bra
%%%
data.mouse.fVtis.int = T.initialize.tissueDB.NaN;
data.mouse.fVtis.int(adi) = 0.135;
data.mouse.fVtis.int(bon) = 0.280/1.500; 
data.mouse.fVtis.int(bra) = 0.004;       
data.mouse.fVtis.int(gut) = 0.600/3.450; 
data.mouse.fVtis.int(hea) = 0.019/0.133;
data.mouse.fVtis.int(kid) = 0.101/0.298;
data.mouse.fVtis.int(liv) = 0.190/0.951;
data.mouse.fVtis.int(lun) = 0.057/0.191;
data.mouse.fVtis.int(mus) = 1.032/7.924;
data.mouse.fVtis.int(ski) = 0.999/2.940;
data.mouse.fVtis.int(spl) = 0.020/0.100;
%%%
data.mouse.fVtis.cel = 1-data.mouse.fVtis.vas-data.mouse.fVtis.int;
%%%
%%% Determining fraction of interstitial and intra-cellular space with
%%% respect to tissue weight NOT INCLUDING vascular blood so that
%%% fVtis.int+fVtis.cel = 1, hence
%%% Scaling: fraction of tissue+vascular space --> fraction of tissue space
%%%
data.mouse.fVtis.int = data.mouse.fVtis.int./(1-data.mouse.fVtis.vas);
data.mouse.fVtis.cel = data.mouse.fVtis.cel./(1-data.mouse.fVtis.vas);


%%% ========================================================================================================
%%% Data  : Regional blood volumes as percentage of total blood volume
%%%
%%% Unit  : percent (%)
%%% Source: 
%%%

allTis = refModel.T.allTis; m = refModel.T.allTisExBlo;  
for a = 1:length(mouse.subtype)   
    
    subtype = mouse.subtype{a};
    mouse.(subtype).fvB   = T.initialize.tissueDB.NaN;

    %%% determine peripheral vascular blood vasB and the fraction of blood 
    %%% not associated with any tissue
    vasB      = T.initialize.tissueDB.NaN;
    vasB(m)   = data.mouse.fVtis.vas(m).*mouse.(subtype).OW(m);
    vasB(blo) = mouse.(subtype).OW(blo)-sum(vasB(m));

    %%% maintain volume ratio between artery and vein compartment
    ratio = mouse.(subtype).OW(ven)/sum(mouse.(subtype).OW([art ven]));
    vasB(ven) =    ratio *vasB(blo);
    vasB(art) = (1-ratio)*vasB(blo);
    
    mouse.(subtype).fvB         = T.initialize.tissueDB.NaN;
    mouse.(subtype).fvB(allTis) = vasB(allTis)/mouse.(subtype).OW(blo);
    mouse.(subtype).fvB(blo)    = mouse.(subtype).fvB(art)+mouse.(subtype).fvB(ven);

end;


%%% ========================================================================================================
%%% Data  : regional blood flows
%%%
%%% Unit  : fraction of co (in L/min)
%%% Source: Brown et al, Tox Ind Health 1997 (Table 24) or for the 3rd column
%%%         Baxter et al. (1994) as fraction of plasma flow = fraction of 
%%%         blood flow; and El-Masri (1998) for adi and bra
%%%
for a = 1:length(mouse.subtype)   
    
    subtype = mouse.subtype{a};
    %%% Brown et al, Tox Ind Health 1997, p.440, however, data based
    %%% approach is belived to be more accurate, BW in [kg]
    mouse.(subtype).co = 0.275*(mouse.(subtype).BW^0.75); 
end;
mouse.mouseFaH.co = 4.38/(1-0.45)/1000; % converting from plasma co in [mL/min] to blood co in [L/min]
mouse.mouseFaH.co = 4.38/(1-0.45)/1000; % converting from plasma co in [mL/min] to blood co in [L/min]

%%%                       Brown     Brown    Baxter   Fronton/Hui
data.mouse.fQco.adi = [ NaN/100,  NaN/100, 7/100    , 0.307/4.380];  % 7% of blood flow, El-Masri (1998) 
data.mouse.fQco.bon = [ NaN/100,  NaN/100, 0.17/4.38, 0.170/4.380];  
data.mouse.fQco.bra = [ 3.3/100,  3.3/100, 3.3/100  ,   3.3/100  ];  % 3.3% of blood flow, El-Masri (1998) 
data.mouse.fQco.gut = [ NaN/100,  NaN/100, 0.90/4.38, 0.900/4.380]; 
data.mouse.fQco.hea = [ 6.6/100,  6.6/100, 0.28/4.38, 0.280/4.380];
data.mouse.fQco.kid = [ 9.1/100,  9.1/100, 0.80/4.38, 0.800/4.380];
data.mouse.fQco.liv = [16.2/100, 16.2/100, 1.10/4.38, 1.100/4.380];
data.mouse.fQco.mus = [15.9/100, 15.9/100, 0.80/4.38, 0.800/4.380];
data.mouse.fQco.ski = [ 5.8/100,  5.8/100, 1.21/4.38, 1.210/4.380];
data.mouse.fQco.spl = [ NaN/100,  NaN/100, 0.05/4.38, 0.050/4.380]; 

for a = 1:length(mouse.subtype)   
    
    subtype = mouse.subtype{a};
    mouse.(subtype).fQco = T.initialize.tissueDB.NaN;
    
    mouse.(subtype).fQco(adi) = data.mouse.fQco.adi(a);
    mouse.(subtype).fQco(bon) = data.mouse.fQco.bon(a);
    mouse.(subtype).fQco(bra) = data.mouse.fQco.bra(a);
    mouse.(subtype).fQco(gut) = data.mouse.fQco.gut(a);
    mouse.(subtype).fQco(hea) = data.mouse.fQco.hea(a);
    mouse.(subtype).fQco(kid) = data.mouse.fQco.kid(a);
    mouse.(subtype).fQco(liv) = data.mouse.fQco.liv(a);
    mouse.(subtype).fQco(mus) = data.mouse.fQco.mus(a);
    mouse.(subtype).fQco(ski) = data.mouse.fQco.ski(a);
    mouse.(subtype).fQco(spl) = data.mouse.fQco.spl(a);
    
    mouse.(subtype).fQco(rob) = 1-sum(mouse.(subtype).fQco(refModel.T.intoVen));
    mouse.(subtype).fQco(lun) = 1-mouse.(subtype).fQco(rob);
    mouse.(subtype).fQco(art) = mouse.(subtype).fQco(lun);
    mouse.(subtype).fQco(ven) = mouse.(subtype).fQco(lun);
    
    
    mouse.(subtype).Q.blo = mouse.(subtype).fQco * mouse.(subtype).co;
end;




%%% ========================================================================================================
%%% Data  : Total tissue water (Vwt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: 
%%%
data.mouse.fVtis.wex = T.initialize.tissueDB.NaN;  %%% tissue water (extra-cellular)
%%%
data.mouse.fVtis.wic = T.initialize.tissueDB.NaN;  %%% tissue water (intra-cellular)


%%% ========================================================================================================
%%% Data  : Neutral lipids (fVtis.nlt) and neutal phospholipids (fVtis.npt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: 
%%%
data.mouse.fVtis.nlt = T.initialize.tissueDB.NaN;  %%% neutral lipids
%%%
data.mouse.fVtis.npt = T.initialize.tissueDB.NaN;  %%% neutral phospholipids 


%%% ========================================================================================================
%%% Data  : Intra-cellular acidic phospholipids (APmt) in rat
%%%
%%% Unit  : mg/g tissue (original) scalled to fraction
%%% Source: 
%%%
data.mouse.fVtis.APmt = T.initialize.tissueDB.NaN;


%%% ========================================================================================================
%%% Data  : Albumin (Atp) in rat
%%%
%%% Unit  : Tissue-to-plasma ratio
%%% Source: 
%%% 
data.mouse.r.Atp  = T.initialize.tissueDB.NaN;


%%% ========================================================================================================
%%% Data  : Lipoprotein (Ltp) in rat
%%%
%%% Unit  : Tissue-to-plasma ratio
%%% Source: 
%%% 
data.mouse.r.Ltp = T.initialize.tissueDB.NaN;

%%% ========================================================================================================
%%% Model : Assignment of fractional volumes and ratio to all age classes
%%%
for a = 1:length(mouse.subtype)
    
    subtype = mouse.subtype{a};
    mouse.(subtype).fVtis = data.mouse.fVtis; 
    mouse.(subtype).r     = data.mouse.r; 
    
end;


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++++++ 
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN + 
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% ========================================================================================================
%%% Model: Age and sex
%%%
%%% Children:
%%% newborn (nb), age 1 (age1), 5 (age5), 10 (age10) , all uni sex
%%% age 15 male (age15m), age 15 female (age15f), 
%%% Adults:
%%% age 20-50 male (age35m), age 20-50 female (age35f)
%%% As in the source, we associate an average age of 35 with the adult
%%% 
%%% Source: ICRP report 2002
%%%
human.type = 'human';

human0 = 1; human1 = 2; 
human.subtype = {'human0','human1','human5','human10','human15m','human15f','human35m','human35f'};


%%% ========================================================================================================
%%% Data  : Body weight (BW), body height (BH) and body surface area (BSA)
%%%
%%% Units : BW in kg, BH in cm, BSA in m^2
%%% Source: ICRP report 2002 (Table 2.9)
%%%
%%%                 nb   age1    age5   age10  age15m  age15f  age35m   age35f
ICRP.BW  =      [ 3.50  10.00   19.00   32.00   56.00   53.00   73.00   60.00];
ICRP.BH  =      [51.00  76.00  109.00  138.00  167.00  161.00  176.00  163.00];
ICRP.BSA =      [ 0.24   0.48    0.78    1.12    1.62    1.55    1.90    1.66];
ICRP.sex =      {'uni'  'uni'   'uni'   'uni'  'male' 'female' 'male' 'female'};
ICRP.age =      [    0      1       5      10     15       15      35      35];

%%% Assign values to all age/sex classes
for a = 1:length(human.subtype)   
    
    age = human.subtype{a};
    human.(age)         = struct(data.subtype);
    human.(age).type    = 'human';
    human.(age).subtype = age;
    human.(age).info    = info;
    human.(age).T       = T;
    human.(age).age     = ICRP.age(a); 
    human.(age).sex     = ICRP.sex{a};
    human.(age).BW      = ICRP.BW(a);    
    human.(age).BH      = ICRP.BH(a)/100;
    human.(age).BSA     = ICRP.BSA(a);   
    human.(age).BMI     = human.(age).BW/human.(age).BH^2;
     
end;


%%% ========================================================================================================
%%% Data  : Organ weight, density and volume 
%%%
%%% Unit  : weight in g, density in g/mL, volume in L
%%% Source: ICRP report 2002 (Table 2.8)
%%% 
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f  
ICRP.OW.adipose_tissue =                       [930   3800   5500   8600   12000   18700   18200   22500]; % entry duplicates other mass information
ICRP.OW.separable_adipose =                    [890   3600   5000   7500    9500   16000   14500   19000];
ICRP.OW.adrenals =                             [  6      4      5      7      10       9      14      13];
% alimentary system
ICRP.OW.tongue =                               [  3.5   10     19     32      56      53      73      60];
ICRP.OW.salivary_glands =                      [  6     24     34     44      68      65      85      70];
ICRP.OW.oesophagus_wall =                      [  2      5     10     18      30      30      40      35];
ICRP.OW.stomach_wall =                         [  7     20     50     85     120     120     150     140];
ICRP.OW.stomach_contents =                     [ 40     67     83    117     200     200     250     230];
ICRP.OW.small_intestine_wall =                 [ 30     85    220    370     520     520     650     600];
ICRP.OW.small_intestine_contents =             [ 56     93    117    163     280     280     350     280];
ICRP.OW.large_intestine_right_colon_wall =     [  7     20     49     85     122     122     150     145];
ICRP.OW.large_intestine_right_colon_contents = [ 24     40     50     70     120     120     150     160];
ICRP.OW.large_intestine_left_colon_wall =      [  7     20     49     85     122     122     150     145];
ICRP.OW.large_intestine_left_colon_contents =  [ 12     20     25     35      60      60      75      80];
ICRP.OW.large_intestine_rectosigmoid_wall =    [  3     10     22     40      56      56      70      70];
ICRP.OW.large_intestine_rectosigmoid_content = [ 12     20     25     35      60      60      75      80];
ICRP.OW.liver =                                [130    330    570    830    1300    1300    1800    1400];
ICRP.OW.gallbladder_wall =                     [  0.5    1.4    2.6    4.4     7.7     7.3    10       8];
ICRP.OW.gallbladder_content =                  [  2.8    8     15     26      45      42      58      48];
ICRP.OW.pancreas =                             [  6     20     35     60     110     100     140     120];

%%% Note: brain organ weight age5  = 1245 = mean value of 1310 (male) and 1180 (female) 
%%%       brain organ weight age10 = 1310 = mean value of 1400 (male) and 1220 (female) 
ICRP.OW.brain =                                [380    950   1245   1310    1420    1300    1450    1300]; 
ICRP.OW.breasts =                              [NaN    NaN    NaN    NaN      15     250      25     500];
% circulatory system
ICRP.OW.heart_with_blood =                     [ 46     98    220    370     660     540     840     620]; % entry duplicates other mass information
ICRP.OW.heart_tissue_only =                    [ 20     50     85    140     230     220     330     250];
% Table sec 7.4
ICRP.OW.total_blood =                          [270    500   1400   2400    4500    3300    5300    3900];
ICRP.OW.eyes =                                 [  6      7     11     12      13      13      15      15];
ICRP.OW.storage_fat =                          [370   2300   3600   6000    9000   14000   14600   18000]; % entry duplicates other mass information
ICRP.OW.skin =                                 [175    350    570    820    2000    1700    3300    2300];
ICRP.OW.skeletal_muscle =                      [800   1900   5600  11000   24000   17000   29000   17500];
ICRP.OW.pituitary_gland =                      [  0.1   0.15   0.25   0.35     0.5     0.5     0.6     0.6];
% Respiratory system
ICRP.OW.larynx =                               [  1.3    4      7     12      22      15      28      19];
ICRP.OW.trachea =                              [  0.5    1.5    2.5    4.5     7.5     6      10       8];
ICRP.OW.lung_with_blood =                      [ 60    150    300    500     900     750    1200     950]; % entry duplicates other mass information
ICRP.OW.lung_tissue_only =                     [ 30     80    125    210     330     290     500     420];
% skeletal system
ICRP.OW.total_skeleton =                       [370   1170   2430   4500    7950    7180   10500    7800]; % entry duplicates other mass information
ICRP.OW.bone_cortical =                        [135    470   1010   1840    3240    2960    4400    3200];
ICRP.OW.bone_trabecular =                      [ 35    120    250    460     810     740    1100     800];
% bone_total = bone_cortical + bone_trabecular
ICRP.OW.bone_total =                           [170    590   1260   2300    4050    3700    5500    4000]; % entry duplicates other mass information
ICRP.OW.marrow_active =                        [ 50    150    340    630    1080    1000    1170     900];
ICRP.OW.marrow_inactive =                      [  0     20    160    630    1480    1380    2480    1800];
ICRP.OW.cartilage =                            [130    360    600    820    1140     920    1100     900];
ICRP.OW.teeth =                                [  0.7    5     15     30      45      35      50      40];
ICRP.OW.skeleton_miscellaneous =               [ 20     45     55     90     155     145     200     160];
ICRP.OW.spleen =                               [  9.5   29     50     80     130     130     150     130];


%%% Note: thymus organ weight age10 = 37.5 = mean value of 40 (male) and 35 (female) 
ICRP.OW.thymus =                               [ 13     30     30     37.5    35      30      25      20];
ICRP.OW.thyroid =                              [  1.3    1.8    3.4    7.9    12      12      20      17];
ICRP.OW.tonsils =                              [  0.1    0.5    2      3       3       3        3      3];
% Urogenital system
ICRP.OW.kidneys =                              [ 25     70    110    180     250     240     310     275];
ICRP.OW.ureters =                              [  0.77   2.2    4.2    7.0    12      12      16      15];
ICRP.OW.urinary_bladder =                      [  4      9     16     25      40      35      50      40];

%%% Note: urethra organ weight nb    = 0.31 = mean value of 0.48 (male) and 0.14 (female) 
%%%       urethra organ weight age1  = 0.91 = mean value of 1.4  (male) and 0.42 (female) 
%%%       urethra organ weight age5  = 1.69 = mean value of 2.6  (male) and 0.78 (female) 
%%%       urethra organ weight age10 = 2.85 = mean value of 4.4  (male) and 1.3  (female) 
ICRP.OW.urethra =                              [  0.31   0.91   1.69   2.85    7.7     2.3    10       3];
ICRP.OW.testes =                               [  0.85   1.5    1.7    2      16       0      35       0];
ICRP.OW.epididymes =                           [  0.25   0.35   0.45   0.6     1.6     0       4       0];
ICRP.OW.prostate =                             [  0.8    1.0    1.2    1.6     4.3     0      17       0];
ICRP.OW.ovaries =                              [  0.3    0.8    2.0    3.5     0       6       0      11];
ICRP.OW.fallopian_tubes =                      [  0.25   0.25   0.35   0.50    0       1.1     0     2.1];
ICRP.OW.uterus =                               [  4.0    1.5    3      4       0      30       0      80];


%%% ========================================================================================================
%%% Data  : Regional blood volumes as percentage of total blood volume
%%%
%%% Unit  : percent (%)
%%% Source: ICRP report 2002 (Table 2.14)
%%%
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f  
ICRP.fVblood.fat =                             [NaN    NaN    NaN    NaN     NaN     NaN    5.00    8.50];
ICRP.fVblood.brain =                           [NaN    NaN    NaN    NaN     NaN     NaN    1.20    1.20];
ICRP.fVblood.stomach_and_oesophages =          [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00];
ICRP.fVblood.small_intestine =                 [NaN    NaN    NaN    NaN     NaN     NaN    3.80    3.80];
ICRP.fVblood.large_intestine =                 [NaN    NaN    NaN    NaN     NaN     NaN    2.20    2.20];
ICRP.fVblood.right_heart =                     [NaN    NaN    NaN    NaN     NaN     NaN    4.50    4.50];
ICRP.fVblood.left_heart =                      [NaN    NaN    NaN    NaN     NaN     NaN    4.50    4.50];
ICRP.fVblood.coronary_tissue =                 [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00];
ICRP.fVblood.kidneys =                         [NaN    NaN    NaN    NaN     NaN     NaN    2.00    2.00];
ICRP.fVblood.liver =                           [NaN    NaN    NaN    NaN     NaN     NaN   10.00   10.00];
ICRP.fVblood.bronchial_tissue =                [NaN    NaN    NaN    NaN     NaN     NaN    2.00    2.00];
ICRP.fVblood.skeletal_muscle =                 [NaN    NaN    NaN    NaN     NaN     NaN   14.00   10.50];
ICRP.fVblood.pancreas =                        [NaN    NaN    NaN    NaN     NaN     NaN    0.60    0.60];
ICRP.fVblood.skeleton.total =                  [NaN    NaN    NaN    NaN     NaN     NaN    7.00    7.00];
ICRP.fVblood.skeleton.red_marrow =             [NaN    NaN    NaN    NaN     NaN     NaN    4.00    4.00]; % entry duplicates other information
ICRP.fVblood.skeleton.trabecular_bone =        [NaN    NaN    NaN    NaN     NaN     NaN    1.20    1.20]; % entry duplicates other information
ICRP.fVblood.skeleton.cortical_bone =          [NaN    NaN    NaN    NaN     NaN     NaN    0.80    0.80]; % entry duplicates other information
ICRP.fVblood.skeleton.others =                 [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00]; % entry duplicates other information
ICRP.fVblood.skin =                            [NaN    NaN    NaN    NaN     NaN     NaN    3.00    3.00];
ICRP.fVblood.spleen =                          [NaN    NaN    NaN    NaN     NaN     NaN    1.40    1.40];
ICRP.fVblood.thyroid =                         [NaN    NaN    NaN    NaN     NaN     NaN    0.06    0.06];
ICRP.fVblood.lymph_nodes =                     [NaN    NaN    NaN    NaN     NaN     NaN    0.20    0.20];
ICRP.fVblood.gonads =                          [NaN    NaN    NaN    NaN     NaN     NaN    0.04    0.02];
ICRP.fVblood.adrenals =                        [NaN    NaN    NaN    NaN     NaN     NaN    0.06    0.06];
ICRP.fVblood.urinary_bladder =                 [NaN    NaN    NaN    NaN     NaN     NaN    0.02    0.02];
ICRP.fVblood.all_other_tissues =               [NaN    NaN    NaN    NaN     NaN     NaN    1.92    1.92];


%%% ========================================================================================================
%%% Data  : Volume of blood plasma and red blood cells
%%%
%%% Unit  : ml
%%% Source: ICRP report (Table 2.12)
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.erythrocyte_volume =                      [NaN    NaN    NaN    NaN     NaN     NaN    2300    1500];
ICRP.plasma_volume =                           [NaN    NaN    NaN    NaN     NaN     NaN    3000    2400];


%%% ========================================================================================================
%%% Data  : Distribution of blood in the vascular system
%%%
%%% Unit  : Percentage (%) of total blood volume
%%% Source: ICRP report 2002 (Table 2.13) 
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.fVblood.heart_chambers =                  [NaN    NaN    NaN    NaN     NaN     NaN    9.00     NaN];
ICRP.fVblood.pulmonary.total =                 [NaN    NaN    NaN    NaN     NaN     NaN   10.50     NaN];
ICRP.fVblood.pulmonary.arteries =              [NaN    NaN    NaN    NaN     NaN     NaN    3.00     NaN];
ICRP.fVblood.pulmonary.capillaries =           [NaN    NaN    NaN    NaN     NaN     NaN    2.00     NaN];
ICRP.fVblood.pulmonary.veins =                 [NaN    NaN    NaN    NaN     NaN     NaN    5.50     NaN];
ICRP.fVblood.systemic.total =                  [NaN    NaN    NaN    NaN     NaN     NaN   80.50     NaN];
ICRP.fVblood.systemic.aorta_large_arteries =   [NaN    NaN    NaN    NaN     NaN     NaN    6.00     NaN];
ICRP.fVblood.systemic.small_arteries =         [NaN    NaN    NaN    NaN     NaN     NaN   10.00     NaN];
ICRP.fVblood.systemic.capillaries =            [NaN    NaN    NaN    NaN     NaN     NaN    5.00     NaN];
ICRP.fVblood.systemic.small_veins =            [NaN    NaN    NaN    NaN     NaN     NaN   41.50     NaN];
ICRP.fVblood.systemic.large_veins =            [NaN    NaN    NaN    NaN     NaN     NaN   18.00     NaN];


%%% ========================================================================================================
%%% Data  : Density of tissue
%%%
%%% Unit  : kg/L (= g/cm^3) = g/mL
%%% Source: Brown 1997 and ICRP report 1975
%%%  
data.human.dens      = ones(size(T.initialize.tissueDB.NaN)); % all organs except for adi and bon (see Brown et al, Tox Ind Health 1997, Table 19)
data.human.dens(adi) = 0.916;                  % ICRP report 1975, p.44
data.human.dens(bon) = 1.3;                    % ICRP report 2002, Table 2.20 (whole skeleton, adults)


%%% ========================================================================================================
%%% Model: Vascular tissue volumes (fraction vascular blood = fvB)
%%%
%%% Unit : fraction of total blood
%%%
for a = 1:length(human.subtype)   
    
    age = human.subtype{a};
    human.(age).fvB      = T.initialize.tissueDB.NaN;
    human.(age).fvB(adi) = ICRP.fVblood.fat(a);
    human.(age).fvB(bon) = ICRP.fVblood.skeleton.total(a);
    human.(age).fvB(bra) = ICRP.fVblood.brain(a);
    human.(age).fvB(gut) = ICRP.fVblood.small_intestine(a)+ICRP.fVblood.large_intestine(a);
    human.(age).fvB(hea) = ICRP.fVblood.coronary_tissue(a);
    human.(age).fvB(kid) = ICRP.fVblood.kidneys(a);
    human.(age).fvB(liv) = ICRP.fVblood.liver(a);
    human.(age).fvB(lun) = ICRP.fVblood.bronchial_tissue(a);
    human.(age).fvB(mus) = ICRP.fVblood.skeletal_muscle(a);
    human.(age).fvB(ski) = ICRP.fVblood.skin(a);
    human.(age).fvB(spl) = ICRP.fVblood.spleen(a);
    
    human.(age).fvB(blo) = 100 - sum(human.(age).fvB(refModel.T.allTisExBlo));
    human.(age).fvB(art) = (0.5*ICRP.fVblood.heart_chambers(a) +...
                           ICRP.fVblood.pulmonary.veins(a) + ...
                           0.5*ICRP.fVblood.pulmonary.capillaries(a) +...
                           ICRP.fVblood.systemic.aorta_large_arteries(a) + ...
                           0.5*ICRP.fVblood.systemic.capillaries(a) + ...
                           ICRP.fVblood.systemic.small_arteries(a))/100 * human.(age).fvB(blo);
    human.(age).fvB(ven) = (0.5*ICRP.fVblood.heart_chambers(a) + ...
                           ICRP.fVblood.pulmonary.arteries(a) +...
                           0.5*ICRP.fVblood.pulmonary.capillaries(a) + ...
                           0.5*ICRP.fVblood.systemic.capillaries(a) +...
                           ICRP.fVblood.systemic.small_veins(a) + ...
                           ICRP.fVblood.systemic.large_veins(a))/100 * human.(age).fvB(blo);
                       
    human.(age).fvB = human.(age).fvB/100; % change of units: percentage -> fraction

    
end;
%%%
%%% DUE TO LACK OF DATA: 
%%%
%%% For children age classes: age35m data were adopted for all fvB entries
%%% except for age15fm, where age35f data were taken
%%% For age35f:age35m data were adopted for fvB values of art and ven 
%%%
human.('human35f').fvB([art ven]) = human.('human35m').fvB([art ven])./human.('human35m').fvB(blo)*...
                                  human.('human35f').fvB(blo);
human.('human0').fvB = human.('human35m').fvB;
human.('human1').fvB    = human.('human35m').fvB;
human.('human5').fvB    = human.('human35m').fvB;
human.('human10').fvB   = human.('human35m').fvB;
human.('human15m').fvB  = human.('human35m').fvB;
human.('human15f').fvB  = human.('human35f').fvB;


%%% ========================================================================================================
%%% Model: Organ weights (OW) and tissue volumes (V)
%%%
%%% Unit : OW in kg and V in L 
%%% NOTE : total blood V(blo) is partitioned into artery blood V(art) and
%%% venous blood V(ven) according to fvB(art)/fvB(blo) and
%%% fvB(ven)/fvB(blo). I.e., all vascular spaces fvB(tis) are partitioned
%%% into an artery and venous part according to fvB(art)/fvB(blo) and
%%% fvB(ven)/fvB(blo)
%%%
for a = 1:length(human.subtype)   
    
    age = human.subtype{a};
    human.(age).OW      = T.initialize.tissueDB.NaN; 
    human.(age).OW(adi) = ICRP.OW.separable_adipose(a);
    human.(age).OW(bon) = ICRP.OW.total_skeleton(a); % includes total bone and marrow
    human.(age).OW(bra) = ICRP.OW.brain(a);
    human.(age).OW(gut) = ICRP.OW.small_intestine_wall(a) + ICRP.OW.large_intestine_right_colon_wall(a) + ...
                          ICRP.OW.large_intestine_left_colon_wall(a) + ICRP.OW.large_intestine_rectosigmoid_wall(a);
    human.(age).OW(hea) = ICRP.OW.heart_tissue_only(a);
    human.(age).OW(kid) = ICRP.OW.kidneys(a);
    human.(age).OW(liv) = ICRP.OW.liver(a);
    human.(age).OW(lun) = ICRP.OW.lung_tissue_only(a);
    human.(age).OW(mus) = ICRP.OW.skeletal_muscle(a);
    human.(age).OW(ski) = ICRP.OW.skin(a);
    human.(age).OW(spl) = ICRP.OW.spleen(a);
    human.(age).OW(blo) = ICRP.OW.total_blood(a);
    human.(age).OW(art) = (human.(age).fvB(art)/human.(age).fvB(blo)) *human.(age).OW(blo);
    human.(age).OW(ven) = (human.(age).fvB(ven)/human.(age).fvB(blo)) *human.(age).OW(blo);
    
    human.(age).OW      = human.(age).OW/1000; % change of units: g -> kg
        
    human.(age).OW(rob) = human.(age).BW - sum(human.(age).OW(refModel.T.allTis));

    human.(age).hct     = ICRP.erythrocyte_volume(a)/ICRP.OW.total_blood(a); 
    human.(age).OW(pla) = (1-human.(age).hct)*human.(age).OW(blo);
    human.(age).OW(ery) =  human.(age).hct*human.(age).OW(blo);
    
    human.(age).fOW     = human.(age).OW/human.(age).BW;
    human.(age).dens    = data.human.dens;
    human.(age).LBW     = human.(age).BW - human.(age).OW(adi); 
    
    human.(age).V.tis   = human.(age).OW./data.human.dens; 
    human.(age).V.vas   = human.(age).fvB*human.(age).V.tis(blo);
    
    human.(age).V.tot   = T.initialize.tissueDB.NaN;
    m = refModel.T.allTisExBlo;
    human.(age).V.tot(m)= human.(age).V.vas(m) + human.(age).V.tis(m);
    m = [blo ven art];
    human.(age).V.tot(m)= human.(age).V.vas(m);
    
end;
%%%
%%% DUE TO LACK OF DATA: 
%%%
%%% For all children age classes: human35m data were adopted for
%%% hematocrit value hct
%%%
%%% Age 15, male and age 10, 5, 1, human0
for subtype = {'human0','human1','human5','human10','human15f'}
    age = char(subtype);
    hct = human.('human35f').hct;      % set identical to hct of female adult
    human.(age).hct = hct;        % in line with NHANES study (for age 5 and older)
    human.(age).V.tis(ery) = hct * human.(age).V.tis(blo);
    human.(age).V.tis(pla) = (1-hct) * human.(age).V.tis(blo);
end;
%%% Age 15, male
hct = human.('human35m').hct;        % set identical to hct of male adult
human.('human15m').hct = hct;     % in line with NHANES study
human.('human15m').V.tis(ery) = hct * human.(age).V.tis(blo);
human.('human15m').V.tis(pla) = (1-hct) * human.(age).V.tis(blo);


%%% ========================================================================================================
%%% Data  : Cardiac output
%%%
%%% Unit  : L/min
%%% Source: ICRP report 2002 (Table 2.39)
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.co  =                                     [NaN    NaN    3.4    5.0     6.1     6.1     6.5     5.9];
ICRP.co([human0 human1]) =                     [44     109]/60; % Alverson et al 1987 (cited in Abraham et al 2005)
 

%%% ========================================================================================================
%%% Data  : blood flow rates
%%%
%%% Unit  : percentage (%) of cardiac output
%%% Source: ICRP report 2002 (Table 2.40)
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.fQco.fat =                                [NaN    NaN    NaN    NaN     NaN     NaN    5.00    8.50];
ICRP.fQco.brain =                              [NaN    NaN    NaN    NaN     NaN     NaN   12.00   12.00];
ICRP.fQco.stomach_and_oesophagus =             [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00]; % not used in PBPK model
ICRP.fQco.small_intestine =                    [NaN    NaN    NaN    NaN     NaN     NaN   10.00   11.00];
ICRP.fQco.large_intestine =                    [NaN    NaN    NaN    NaN     NaN     NaN    4.00    5.00];
ICRP.fQco.coronary_tissue =                    [NaN    NaN    NaN    NaN     NaN     NaN    4.00    5.00];
ICRP.fQco.kidney =                             [NaN    NaN    NaN    NaN     NaN     NaN   19.00   17.00];
ICRP.fQco.liver =                              [NaN    NaN    NaN    NaN     NaN     NaN   25.50   27.00]; % total, arterial = [6.5 6.5]
ICRP.fQco.bronchial_tissue =                   [NaN    NaN    NaN    NaN     NaN     NaN    2.50    2.50]; % not used in PBPK model
ICRP.fQco.skeletal_muscle =                    [NaN    NaN    NaN    NaN     NaN     NaN   17.00   12.00];
ICRP.fQco.pancreas =                           [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00]; % not used in PBPK model
ICRP.fQco.skeleton.total =                     [NaN    NaN    NaN    NaN     NaN     NaN    5.00    5.00];
ICRP.fQco.skeleton.red_marrow =                [NaN    NaN    NaN    NaN     NaN     NaN    3.00    3.00];
ICRP.fQco.skeleton.trabecular_bone =           [NaN    NaN    NaN    NaN     NaN     NaN    0.90    0.90];
ICRP.fQco.skeleton.cortical_bone =             [NaN    NaN    NaN    NaN     NaN     NaN    0.60    0.60];
ICRP.fQco.skeleton.others =                    [NaN    NaN    NaN    NaN     NaN     NaN    0.50    0.50];
ICRP.fQco.skin =                               [NaN    NaN    NaN    NaN     NaN     NaN    5.00    5.00];
ICRP.fQco.spleen =                             [NaN    NaN    NaN    NaN     NaN     NaN    3.00    3.00];
ICRP.fQco.thyroid =                            [NaN    NaN    NaN    NaN     NaN     NaN    1.50    1.50]; % not used in PBPK model
ICRP.fQco.lymph_nodes =                        [NaN    NaN    NaN    NaN     NaN     NaN    1.70    1.70]; % not used in PBPK model
ICRP.fQco.gonads =                             [NaN    NaN    NaN    NaN     NaN     NaN    0.05    0.02]; % not used in PBPK model
ICRP.fQco.adrenals =                           [NaN    NaN    NaN    NaN     NaN     NaN    0.30    0.30]; % not used in PBPK model
ICRP.fQco.urinary_bladder =                    [NaN    NaN    NaN    NaN     NaN     NaN    0.06    0.06]; % not used in PBPK model
ICRP.fQco.all_other_tissues =                  [NaN    NaN    NaN    NaN     NaN     NaN    1.39    1.92]; % not used in PBPK model 


%%% ========================================================================================================
%%% Model: Tissue blood flow (fraction of cardiac output and absolut values)
%%%
%%% Unit : fraction and L/min
%%% 
%%% 
for a = 1:length(human.subtype)   
    
    age = human.subtype{a};
    human.(age).fQco      = T.initialize.tissueDB.NaN; 
    human.(age).Q.blo     = T.initialize.tissueDB.NaN; 
    human.(age).fQco(adi) = ICRP.fQco.fat(a);
    human.(age).fQco(bon) = ICRP.fQco.skeleton.total(a);
    human.(age).fQco(bra) = ICRP.fQco.brain(a);
    human.(age).fQco(gut) = ICRP.fQco.small_intestine(a)+ICRP.fQco.large_intestine(a);
    human.(age).fQco(hea) = ICRP.fQco.coronary_tissue(a);
    human.(age).fQco(kid) = ICRP.fQco.kidney(a);
    human.(age).fQco(liv) = ICRP.fQco.liver(a);
    human.(age).fQco(mus) = ICRP.fQco.skeletal_muscle(a);
    human.(age).fQco(ski) = ICRP.fQco.skin(a);
    human.(age).fQco(spl) = ICRP.fQco.spleen(a);

    human.(age).fQco(rob) = 100 - sum(human.(age).fQco(refModel.T.intoVen));
    
    human.(age).fQco(lun) = 100 - human.(age).fQco(rob); % ensures closed blood flow system
    human.(age).fQco(art) = human.(age).fQco(lun);
    human.(age).fQco(ven) = human.(age).fQco(lun);
    
    human.(age).fQco      = human.(age).fQco/100;        % change of units: [%] -> fraction
    
    human.(age).co        = ICRP.co(a); 
    human.(age).Q.blo     = human.(age).fQco*human.(age).co;


end;
%%%
%%% DUE TO LACK OF DATA: 
%%%
%%% For all children age classes: fQco values were estimated based on the
%%% approach presented in Abraham et al, Arch Toxicol (2005) 79: 63?73
%%%

children    = {'human0','human1',  'human5' , 'human10', 'human15m','human15f'};
refAgeClass = {'human35m', 'human35m','human35m','human35m','human35m','human35m'};  

% Note: Scaling 'human15f' based on 'human35f' would result in liver blood of 'human15f' larger than for 'human35f'
Q_perKg_bra = [50 59 71 68 57 57]*10/1000;        % brain: ml/min/100g -> L/min/kg, Chiron et al, 1992

for a = 1:length(children)   
    
    age = children{a};
    refAge = char(refAgeClass(a));
    
    Q_perKg = (human.(refAge).Q.blo) ./ human.(refAge).V.tis;
    Q_hat   = Q_perKg.*human.(age).V.tis;
    
    Q_blo      = T.initialize.tissueDB.NaN;
    Q_blo(kid) = Q_hat(kid);                      % kidneys: Grunert et al, 1990, Q per g tissue independent of age
    Q_blo(bra) = Q_perKg_bra(a)*human.(age).V.tis(bra); 
    
    tis = [refModel.T.intoVen rob];                   % all tissues with infow into the ven, including rob
    tis(find([kid==tis])) = []; tis(find([bra==tis])) = []; % as above, but excluding bra and kid
    normalization_Qco = sum(Q_hat(tis))/(human.(age).co - Q_blo(kid) - Q_blo(bra)); 
    if normalization_Qco >1
        error('Scaling blood flows to children resulted in cardiac output that is larger than the experimental reported one!');
    end;
    Q_blo(tis) = Q_hat(tis)/normalization_Qco;
    
    Q_blo([art lun ven]) = human.(age).co - Q_blo(rob);
    % pv = portal vein
    Q_perKg_hepart = (human.(refAge).Q.blo(liv)-human.(refAge).Q.blo(gut)-human.(refAge).Q.blo(spl))/human.(refAge).V.tis(liv);
    Q_hat_hepart   = Q_perKg_hepart*human.(age).V.tis(liv);
    normalization_Qpv  = (Q_hat_hepart+Q_hat(gut)+Q_hat(spl))/Q_blo(liv);
    if normalization_Qpv >1
        error('Scaling blood flows to children resulted in liver blood flow that is larger than the experimental reported one!');
    end;
    tis = [gut spl];
    Q_blo(tis) = Q_hat(tis)/normalization_Qpv;
    
    human.(age).Q.blo = Q_blo;
    human.(age).fQco = human.(age).Q.blo/human.(age).co;
    
end;


%%% ========================================================================================================
%%% Data  : Total tissue water (Vwt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: Poulin & Theil, J Pharm Sci, Vol 91, 2002 and Poulin & Theil, J Pharm Sci, 2009 for ery value
%%% 
data.human.fVtis.wtot = T.initialize.tissueDB.NaN;
data.human.fVtis.wtot(lun) = 0.811;
data.human.fVtis.wtot(kid) = 0.783;
data.human.fVtis.wtot(spl) = 0.778;
data.human.fVtis.wtot(bra) = 0.77;
data.human.fVtis.wtot(mus) = 0.76;
data.human.fVtis.wtot(hea) = 0.758;
data.human.fVtis.wtot(liv) = 0.751;
data.human.fVtis.wtot(gut) = 0.718;
data.human.fVtis.wtot(ski) = 0.718; 
data.human.fVtis.wtot(bon) = 0.439; 
data.human.fVtis.wtot(adi) = 0.18;
data.human.fVtis.wtot(pla) = 0.945;
data.human.fVtis.wtot(ery) = 0.63;       
%%%
%%% DUE TO LACK OF DATA: 
%%%
%%% Assume that ratio wex-to-wtot is the same as in RAT
%%%
r_wex_to_wtot   = data.rat.fVtis.wex./(data.rat.fVtis.wex+data.rat.fVtis.wic);
%%%
data.human.fVtis.wex = T.initialize.tissueDB.NaN; %%% fraction of extra-celluar water (wex)
data.human.fVtis.wex = r_wex_to_wtot.*data.human.fVtis.wtot;
data.human.fVtis.wex(ery) = 0;
%%%
data.human.fVtis.wic = T.initialize.tissueDB.NaN; %%% fraction of intracellular water (wic)
data.human.fVtis.wic = data.human.fVtis.wtot-data.human.fVtis.wex;


%%% ========================================================================================================
%%% Data  : Fraction of interstitial and intra-cellular space 
%%%
%%% Unit  : fraction 
%%%
data.human.fVtis.int = T.initialize.tissueDB.NaN; 
data.human.fVtis.cel = T.initialize.tissueDB.NaN; 
%%%
%%% Due to LACK OF DATA (assume same values as for rat)
%%%
data.human.fVtis.int = data.rat.fVtis.int;
data.human.fVtis.cel = data.rat.fVtis.cel;



%%% ========================================================================================================
%%% Data  : Neutral lipids (fVtis.nlt) and neutal phospholipids (fVtis.npt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: Poulin and Theil, J Pharm Sci, Vol. 91, 2002
%%%         Rodgers and Rowland, J Pharm Res, Vol. 24, 2007 (erythrocyte
%%%         values only), (Table VII)
%%%
data.human.fVtis.nlt = T.initialize.tissueDB.NaN;  %%% neutral lipids
data.human.fVtis.nlt(adi) = 0.79;
data.human.fVtis.nlt(bon) = 0.074;
data.human.fVtis.nlt(bra) = 0.051;
data.human.fVtis.nlt(gut) = 0.0487;
data.human.fVtis.nlt(hea) = 0.0115;
data.human.fVtis.nlt(kid) = 0.0207;
data.human.fVtis.nlt(liv) = 0.0348;
data.human.fVtis.nlt(lun) = 0.003;
data.human.fVtis.nlt(mus) = 0.0238;  
data.human.fVtis.nlt(ski) = 0.0284;
data.human.fVtis.nlt(spl) = 0.0201;
data.human.fVtis.nlt(pla) = 0.0035;
data.human.fVtis.nlt(ery) = 0.0033; 
%%%
data.human.fVtis.npt = T.initialize.tissueDB.NaN;  %%% neutral phospholipids
data.human.fVtis.npt(bra) = 0.0565;
data.human.fVtis.npt(liv) = 0.0252; 
data.human.fVtis.npt(spl) = 0.0198;
data.human.fVtis.npt(hea) = 0.0166;
data.human.fVtis.npt(gut) = 0.0163;
data.human.fVtis.npt(kid) = 0.0162;
data.human.fVtis.npt(ski) = 0.0111;
data.human.fVtis.npt(lun) = 0.009;
data.human.fVtis.npt(mus) = 0.0072; 
data.human.fVtis.npt(adi) = 0.002;
data.human.fVtis.npt(bon) = 0.0011;
data.human.fVtis.npt(pla) = 0.00225;
data.human.fVtis.npt(ery) = 0.0012;  


%%% ========================================================================================================
%%% Data  : Intra-cellular acidic phospholipids (APmt) 
%%%
%%% Unit  : fraction
%%% DUE TO LACK OF DATA: 
%%%
%%% Rat values adopted for human (see Rodgers, Leahy, and Rowland, J Pharm
%%% Sci 94 (2005), page 925 (right, 2nd paragraph)
%%%
data.human.fVtis.APmt = data.rat.fVtis.APmt;


%%% ========================================================================================================
%%% Data  : Albumin (Atp) 
%%%
%%% Unit  : Tissue-to-plasma ratio
%%% DUE TO LACK OF DATA: 
%%%
%%% Rat values adopted for human (see Rodgers, Leahy, and Rowland, J Pharm
%%% Sci 94 (2005), page 925 (right, 2nd paragraph)
%%%
data.human.r.Atp = data.rat.r.Atp; 


%%% ========================================================================================================
%%% Data  : Lipoprotein (Ltp)
%%%
%%% Unit  : Tissue-to-plasma ratio
%%% DUE TO LACK OF DATA: 
%%%
%%% Rat values adopted for human (see Rodgers, Leahy, and Rowland, J Pharm
%%% Sci 94 (2005), page 925 (right, 2nd paragraph)
%%%
data.human.r.Ltp = data.rat.r.Ltp;


%%% ========================================================================================================
%%% Model : Assignment of fractional volumes and ratio to all age classes
%%%
for a = 1:length(human.subtype)   
    
    subtype = human.subtype{a};
    human.(subtype).r     = data.human.r; 
    human.(subtype).fVtis = data.human.fVtis; 
    
end;


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ 
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



%%% ========================================================================================================
%%% Model : Assignment of species and tissue
%%%

species.type = {'human','rat','mouse'};

species.info     = info;
species.T        = T;

species.rat      = rat;
species.mouse    = mouse;
species.human    = human;

