%%% 
%%% This function specifies a PBPK model for sMD with
%%% well-stirred tissue distribution and hepatic clearance
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_sMD_PBPK_13CMT_wellstirred(individual)

%%% check whether drug class is really sMD
%%%
if ~strcmp(individual.drug.class,'sMD')
    message = sprintf('Model >>sMD_PBPK_13CMTs_wellstirred<< not defined for drug type %s!',individual.drug.class);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

%%% define model specific parameters
%%%
individual = GenericPBPKmodel_defineModelParameters(individual);

%%% simulate PBPK model 
%%%
model = individual.model; S = model.S; study = individual.study;
X0 = model.X0; sim.t = []; sim.X = []; 

for d=1:study.numberOfDosing
    
    %%% account for dosing for each new dosing interval
    X0(S.bolus)   = X0(S.bolus) + study.bolus.dose*model.SF.mg_to_nmol/model.V.iv_bolus;
    X0(S.IVbag)   = study.infusion.dose*model.SF.mg_to_nmol; 
    X0(S.GItract) = X0(S.GItract) + study.po.dose*model.SF.mg_to_nmol;

    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] = ode15s(@GenericPBPKmodel_RHS,study.observationTime,X0',[],individual);
    
    %%% transform relative to absolute time and modify last time point
    %%% to allow for trough value measurement
    Tend  = max(study.observationTime);
    t = t + (d-1)*Tend; 
    if d~=study.numberOfDosing
        t(end)  = t(end) - 1e-10; 
    end;
    
    %%% store current output in sim structure
    sim.t = [sim.t; t];
    sim.X = [sim.X; X];

    X0 = X(end,:)'; 

end;

%%% determine standart simulation output
%%%
individual = GenericPBPKmodel_determineStandartOutput(individual,sim);

%%% graphical output specific to this model
%%%
GenericPBPKmodel_specificGraphicalOutput(individual);

end 
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_defineModelParameters(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model   = individual.model;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

%%% tissue topology indexing
%%%
S = individual.T;

S.allTis      = [S.lun S.art S.adi S.bra S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv S.ven];
S.allTisExBlo = [S.lun       S.adi S.bra S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv      ];
S.artInflow   = [            S.adi S.bra S.hea S.kid S.mus S.bon S.ski S.gut S.spl            ];
S.intoVen     = [            S.adi S.bra S.hea S.kid S.mus S.bon S.ski             S.liv      ];
S.intoLiv     = [                                                      S.gut S.spl            ];
S.bolus       = S.ven;    % i.v. bolus administration 
S.tissueDB    = S.allTis; % specify which tissues are part of the general tissue DB

S.maxIndex.tis = max(S.allTis);
S.initialize.tis.NaN = NaN*ones(1,S.maxIndex.tis);

%%% dosing and metabolism indexing

S.GItract = 1 + S.maxIndex.tis; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.tis; % IVbag (for infusion dosing)
S.metab   = 3 + S.maxIndex.tis; % metabolized drug

S.maxIndex.dosing = 3 + S.maxIndex.tis;

%%% additional indexing
%%%

S.maxIndex.all = 0 + S.maxIndex.dosing;

%%% ODE indexing (for relative indexing)
%%%
S.C_tot   =   1:S.maxIndex.tis; % all tissues


%%% -----------------------------------------------------------------------
%%% Define PBPK parameters 
%%%

a   = S.allTis;
sMD = individual.drug;

%%% tissue volumes 
%%%
V.tis        = S.initialize.tis.NaN;
V.tis(a)     = individual.V.tis(a);
% set V.tis=0 for S.art and S.ven; and account for them with V.vas
V.tis([S.art S.ven]) = 0; 

V.vas        = S.initialize.tis.NaN;
V.vas(a)     = individual.V.vas(a);

V.tot        = V.vas + V.tis;

%%% volume for i.v. bolus dosing
%%%
V.iv_bolus   = V.tot(S.ven);  

%%% total tissue-to-blood partition coefficients
%%%
K.tis_up     = S.initialize.tis.NaN;
K.tis_up(a)  = sMD.K.tis_up(a);

K.tis        = sMD.fuP/sMD.BP*K.tis_up; 
K.tot        = (V.tis.*K.tis + V.vas) ./ V.tot;

%%% intrinsic tissue clearance
%%%
CLint.tis    = S.initialize.tis.NaN;
CLint.tis(a) = sMD.CLint.tis(a);

%%% blood flows (ensure closed circulatory system!)
%%%
Q.blo        = S.initialize.tis.NaN;
Q.blo(a)     = individual.Q.blo(a);
Q.blo([S.ven S.lun S.art]) = sum(Q.blo(S.intoVen));


%%% extraction ratio based on well-stirred tissue model
%%%
E            = sMD.E; % keep E.gut and E.feces
E.tis        = S.initialize.tis.NaN;
E.tis(a)     = 0;
% only liver assumed to be extracting organ
elim         = S.liv; 
E.tis(elim)  = (CLint.tis(elim).*K.tis(elim))./(Q.blo(elim)+CLint.tis(elim).*K.tis(elim));

%%% elimination corrected partition coefficients
eK.tis       = (1-E.tis).*K.tis; 
eK.tot       = (V.tis.*eK.tis + V.vas) ./ V.tot;

%%% define hepatic clearance wrt. joint vascular & tissue concentration
CLint.tot    = CLint.tis.*K.tis./K.tot;

CLblo        = Q.blo.*E.tis;

%%% first order po absorption rate constant
lambda_po     = individual.drug.lambda_po;

%%% initial condition of ODEs
%%%
X0 = zeros(1,S.maxIndex.all);


%%% -----------------------------------------------------------------------
%%% Assign model parameters 
%%%

model.S     = S;
model.V     = V;
model.Q     = Q;
model.K     = K;
model.CLint = CLint;
model.CLblo = CLblo;
model.E     = E;
model.eK    = eK;
model.lambda_po = lambda_po;
model.SF = sMD.SF;
model.X0 = X0;

individual.model = model;
end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = individual.model; S = model.S;

%%% variables (always use row vector notation)
C_tot     = X(S.C_tot)';
A_GItract = X(S.GItract)';

%%% tissue volumes, blood flows, extraction ratios, clearance etc.
V     = model.V.tot;
Q     = model.Q.blo; 
K     = model.K.tot;
CLint = model.CLint.tot;
E     = model.E;
infusion = individual.study.infusion;

infusion_rate = 0;
if (infusion.tend > 0) && (t<=infusion.tend)
    infusion_rate = model.SF.mg_to_nmol * infusion.dose/infusion.tend;
end;
lambda_po = model.lambda_po; 

%%% venous concentration leaving tissue
C_vbl = C_tot./K;

%%% hepatic artery flow, inflowing concentration into liver and vein
Q_hepart = Q(S.liv)-sum(Q(S.intoLiv));
C_in_liv = (sum(Q(S.intoLiv).*C_vbl(S.intoLiv)) + Q_hepart*C_tot(S.art)) / Q(S.liv);
C_in_ven = sum(Q(S.intoVen).*C_vbl(S.intoVen))/Q(S.ven);


%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
VdC_tot = zeros(size(C_tot)); 

%%% lung 
VdC_tot(S.lun) = Q(S.lun)*(C_tot(S.ven) - C_vbl(S.lun));

%%% artery
VdC_tot(S.art) = Q(S.art)*(C_vbl(S.lun) - C_tot(S.art));

%%% all tissues that are supplied by the artery only
tis = S.artInflow; 
VdC_tot(tis)   = Q(tis).*(C_tot(S.art) - C_vbl(tis));

%%% liver
VdC_tot(S.liv) = Q(S.liv)*(C_in_liv - C_vbl(S.liv)) ...
                - CLint(S.liv)*C_tot(S.liv) + (1-E.gut)*(1-E.feces)*lambda_po*A_GItract;
%%% vein
VdC_tot(S.ven) = Q(S.ven)*(C_in_ven - C_tot(S.ven)) + infusion_rate;

%%% drug amount in GItract for absorption
dA_GItract = -lambda_po*A_GItract;

%%% drug amount in IVbag for infusion
dA_IVbag   = -infusion_rate;

%%% drug amount metabolized or excreted
dA_metab   = +CLint(S.liv)*C_tot(S.liv) + (1-(1-E.gut)*(1-E.feces))*lambda_po*A_GItract;

%%%
%%% END OF ODEs 
%%% -----------------------------------------------------------------------


%%% converting amounts to concentrations
dC_tot = zeros(size(C_tot)); 
dC_tot(S.allTis) = VdC_tot(S.allTis)./V(S.allTis);


%%% output vector (always in column vector notation)
dX(S.C_tot)   = dC_tot';
dX(S.GItract) = dA_GItract';
dX(S.IVbag)   = dA_IVbag';
dX(S.metab)   = dA_metab';

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_determineStandartOutput(individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model; info = individual.info; S = model.S; 
pred  = individual.pred; 

%%% assign predicted profiles
%%%
info.pred.t         = 'simulation time in [min]';
pred.t              = sim.t;

%%% initialize all compartments with NaN and assign only to those
%%% compartments values that are part of the model topology
initialize.NaN = ones(size(pred.t))*S.initialize.tis.NaN; tis = S.allTis;

info.pred.C.tot     = 'concentration-time profile in total tissue space [nmol/L]';
pred.C.tot          = initialize.NaN;
pred.C.tot(:,tis)   = sim.X(:,tis);

info.pred.A.tot     = 'amount in total tissue space [nmol]';
pred.A.tot          = initialize.NaN;
pred.A.tot(:,tis)   = pred.C.tot(:,tis)*diag(model.V.tot(tis));

info.pred.A.body    = 'total amount in the body in [nmol]';
pred.A.body         = sum(pred.A.tot(:,tis),2);

info.pred.A.GItract = 'remaining amount in the GI tract in [nmol]';
pred.A.GItract      = sim.X(:,S.GItract);

info.pred.A.IVbag   = 'remaining amount in the IVbag in [nmol]';
pred.A.IVbag        = sim.X(:,S.IVbag);

info.pred.A.metab   = 'amount metabolized in [nmol]';
pred.A.metab        = sim.X(:,S.metab);


%%% determine standard output in [mg/L] or [mg]
%%%
stdout = individual.stdout; T = stdout.T;
initialize.NaN = ones(size(pred.t))*T.initialize.tissueDB.NaN; SF = model.SF; 

info.stdout.t          = 'simulation time in [min]';
stdout.t               = pred.t;

info.stdout.C.tis      = 'concentration-time profile in tissue space [mg/L]';
stdout.C.tis           = initialize.NaN;
for k = intersect(T.tissueDB,S.tissueDB)
    stdout.C.tis(:,k)  = SF.nmol_to_mg*pred.C.tot(:,k)*(model.eK.tis(k)./model.eK.tot(k));
end;
stdout.C.tis(:,T.pla)  = stdout.C.tis(:,T.ven)/individual.drug.BP;

info.stdout.nC.tis     = 'normalized concentration-time profile (nC.tis = C.tis/eK.tis) in tissue space [mg/L] ';
stdout.nC.tis          = initialize.NaN;
for k = intersect(T.tissueDB,S.tissueDB)
    stdout.nC.tis(:,k) = stdout.C.tis(:,k)/model.eK.tis(k);
end;

info.stdout.A.body     = 'total amount in the body in [mg]';
stdout.A.body          = SF.nmol_to_mg*pred.A.body;

info.stdout.A.GItract  = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract       = SF.nmol_to_mg*pred.A.GItract;

info.stdout.A.IVbag    = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag         = SF.nmol_to_mg*pred.A.IVbag;

info.stdout.A.metab    = 'amount metabolized in [mg]';
stdout.A.metab         = SF.nmol_to_mg*pred.A.metab;


individual.info   = info;
individual.pred   = pred;
individual.stdout = stdout;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_specificGraphicalOutput(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

return;

T = individual.T; stdout = individual.stdout; id = individual.id;

if ~ismember(id,[1 6])
    return;
end;

tis = [T.pla T.liv T.kid T.mus];

for k = 1:length(tis) 
    
    C = stdout.C.tis(:,tis(k)); t = stdout.t/60;
    
    figure(70+k); if id==1 clf; end;
    if id==1 || id==6
        hold on;
        semilogy(t,C,individual.color);
        xlabel('t in [h]'); ylabel(sprintf('C in [mg/L] in %s',T.name{tis(k)}));
        hold off;
        
    end;
    set(gca,'YScale','log'); xlim([0 12]); ylim([1e-2 1.1e1]);
    fett(70+k);
    h = findobj(gca,'Type','line'); set(h,'LineWidth',5);
    
end;

% print -depsc -f71 Plasma_human35mhuman5_drugB_146g_po
% print -depsc -f72 Liver_human35mhuman5_drugB_146g_po
% print -depsc -f73 Kidney_human35mhuman5_drugB_146g_po
% print -depsc -f74 Muscle_human35mhuman5_drugB_146g_po

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

