%%% 
%%% This function compiles the reference individuals, combinding anatomical
%%% and physiological data, drug-specific data and study information
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function reference = GenericPBPKmodel_referenceIndividual(mammals,drug,study)

fprintf('\n   (b) Compiling reference animals/human')

%%% initialize data structure
%%%

for s = 1:length(mammals.type) 
    
    species = mammals.(mammals.type{s});
    
    for a = 1:length(species.subtype) % consider all subtypes of a species
        
        subtype = species.subtype{a};
        
        %%% assign physiological and anatomical parameter values
        %%%
        reference.(subtype) = assign_physiology_data(species,subtype);
        
        %%% assign drug specific parameter values
        %%%
        reference.(subtype) = assign_drug_data(reference.(subtype),drug);
        
        %%% assign dosage information
        %%%
        reference.(subtype) = assign_study_data(reference.(subtype),study);
        
        %%% assign default PBPK tissue model = well-stirred model for 
        %%% joint vascular & tissue space (can be changed when generating
        %%% a virtual population
        %%%
        reference.(subtype) = set_default_model(reference.(subtype));
        
        %%% add sub-fields to store simulation results
        %%%
        reference.(subtype).pred = [];

        %%% add sub-fields to store simulation results
        %%%
        reference.(subtype).stdout.T = mammals.T;
        
    end;
end;

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function reference = assign_physiology_data(species,subtype)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% inherit fieldnames (parameter name ,units and info) from subtype structure
%%%
for n = {'type','subtype','T','info','id','color','I','age','sex','BH','BW',...
         'LBW','BMI','BSA','OW','fOW','dens','V','hct','fvB','fVtis','r','co','Q','fQco'}
    
    name = char(n);
    reference.info.(name) = species.(subtype).info.(name);
    reference.(name)      = species.(subtype).(name);  
    
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function reference = assign_drug_data(reference,drug)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% input variable 'reference' is a single reference individual
%%% rather than the structure comprising all reference individuals

T = reference.T;

switch drug.class
    
    %%% -------------------------------------------------------------------
    %%% small molecule drugs
    %%% ------------------------------------------------------------------- 
    case 'sMD';
        
        if ~isfield(drug,reference.type)
            message = sprintf('No %s specific data for drug %s available!',reference.type,drug.name);
            GenerikPBPKmodel_ReportErrorMessage(message);
        end;    
       
        for n = {'name','class','subclass','MW','pKa','logPow','logPvow'}            
            name = char(n);
            sMD.info.(name) = drug.info.(name);
            sMD.(name)      = drug.(name);   
        end;
        
        
        %%% mixed drug-species/subtype related data
        %%%
        
        mixed = drug.(reference.type);
        
        %%% fraction unbound in plasma assumed to be the same for all subtypes of
        %%% a given species
        sMD.fuP  = mixed.fuP;
        
        %%% blood-to-plasma ratio
        hct = reference.hct;
        sMD.K_ery_up = mixed.K_ery_up;
        sMD.BP   = hct*sMD.K_ery_up*sMD.fuP + (1-hct);
        
        %%% tissue-to-unbound partition coefficients according to Rodgers&Rowland
        method.name       = 'RodgersRowland';
        sMD.K.tis_up      = GenericPBPKmodel_TissuePartitionCoefficients(reference,sMD,method);
        sMD.info.K.tis_up = sprintf('tissue-to-unbound plasma partition coefficient (%s)',method.name);
        
        %%% intrinsic tissue clearance
        sMD.CLint.tis      = T.initialize.tissueDB.NaN;
        sMD.info.CLint.tis = 'tissue specific intrinsic clearance in [L/min]';
        
        K_liv_blo = sMD.K.tis_up(T.liv)*sMD.fuP/sMD.BP;
        sMD.CLint.tis(T.liv) = mixed.CLintK_kgOW.liv/K_liv_blo*reference.OW(T.liv);
                
        sMD.E         = mixed.E;         sMD.info.E = drug.info.E;
        sMD.lambda_po = mixed.lambda_po; sMD.info.lambda_po = drug.info.lambda_po;
        
        %%% scaling factor from mg to mirco-mol       
        sMD.SF.mg_to_nmol = 1e6/drug.MW;
        sMD.SF.nmol_to_mg = 1/sMD.SF.mg_to_nmol;

        %%% assign drug specific parameters
        reference.drug = sMD;
        
        
    %%% -------------------------------------------------------------------
    %%% monoclonal antibodies
    %%% -------------------------------------------------------------------
    case 'mAb'
        
        for n = {'name','class','subclass','MW'}            
            name = char(n);
            mAb.info.(name) = drug.info.(name);
            mAb.(name)      = drug.(name);   
        end;

        %%% scaling factor from mg to nano-mol       
        mAb.SF.mg_to_nmol = 1e6/drug.MW;
        mAb.SF.nmol_to_mg = 1/mAb.SF.mg_to_nmol;

        %%% assign drug specific parameters
        reference.drug = mAb;

end;

end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function reference = assign_study_data(reference,study)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Input variable 'reference' is a single reference individual
%%% rather than the structure comprising all reference individuals

%%% dosing
study.route = {'po','bolus','infusion'};
for d = 1:length(study.route)
    
    route  = study.route{d};
    dosing = study.dosing.(route);
    %%% assign fixed dose or compute corresponding individual dose based on
    %%% the BSD
    if  strcmp(dosing.per,'fixed')
        study.(route).dose = dosing.dose;
    else
        BSD  = dosing.per;
        if ~isfield(reference,BSD)
            message = sprintf(' Unknown choice of dose per ''%s'' for %s dosing',BSD,route);
            GenericPBPKmodel_ReportErrorMessage(message);
        end;
        if ~isfinite(reference.(BSD))
            message = sprintf(' %s'' for species %s NaN or infinite',BSD,reference.type);
            GenericPBPKmodel_ReportErrorMessage(message);
        end;
        study.(route).dose = dosing.dose*reference.(BSD);
    end;
    if strcmp(route,'infusion')
        study.infusion.tend = dosing.tend;
    end;
    
end;
reference.study = study;

%%% update info
info = reference.info;
info.study.dosing.bolus.dose      = 'bolus dose in [mg] or [mg/BSD], i.e. scaled per BSD';
info.study.dosing.bolus.per       = 'descriptor to scale the i.v. bolus dose';
info.study.dosing.infusion.dose   = 'dose of i.v. infusion in [mg] or [mg/BSD], i.e. scaled per BSD';
info.study.dosing.infusion.per    = 'descriptor to scale the i.v. infusion dose';
info.study.dosing.infusion.tend   = 'duration of infusion in [min]';
info.study.dosing.po.dose         = 'p.o. dose in [mg] or [mg/BSD], i.e. scaled per BSD';
info.study.dosing.po.per          = 'descriptor to scale the p.o. dose';

info.study.observationTime        = 'time interval of a single dose or between multiple doses in [min]';
info.study.tend                   = 'time between multiple doses in [min], defined as max(observationTime)';
info.study.numberOfDosings        = 'number of dosings';
info.study.bolus.dose             = 'individual dose of i.v. bolus administration in [mg]';
info.study.infusion.dose          = 'individual dose of i.v. infusion in [mg]';
info.study.infusion.tend          = 'duration of i.v. infusion in [min]';
info.study.po.dose                = 'individual dose of p.o. adminisration in [mg]';

reference.info      = info;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = set_default_model(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


switch individual.drug.class
    
    case 'sMD'
        %%% ---------------------------------------------------------------
        %%% Default model: 13 cmt whole-body PBPK model based on the 
        %%%                perfusion-rate limited tissue model 
        %%%                (also termed well-stirred tissue model)
        %%% ---------------------------------------------------------------
        individual.model.type = 'sMD_PBPK_13CMT_wellstirred';  

    case 'mAb'
        %%% ---------------------------------------------------------------
        %%% Default model: 11 cmt whole-body PBPK model based on the 
        %%%                extravasation-rate limed tissue model
        %%% ---------------------------------------------------------------
        
        individual.model.type = 'mAb_PBPK_11CMT_extravasationLimited'; 
end;
 

end


%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



