%%% 
%%% This function specifies a lumped PBPK model for sMD with
%%% well-stirred tissue distribution and hepatic clearance
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_sMD_lumpedPBPK_xCMT_wellstirred(individual)

%%% check whether drug class is really sMD
%%%
if ~strcmp(individual.drug.class,'sMD')
    message('GenericPBPKmodel_sMD_wellstirred_tot_model not defined for drug type %s!',individual.drug.class);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

%%% define model specific parameters
%%%
individual = GenericPBPKmodel_defineModelParameters(individual);

%%% simulate PBPK model 
%%%
model = individual.model; S = model.S; study = individual.study;
X0 = model.X0; sim.t = []; sim.X = []; 

for d=1:study.numberOfDosing
    
    %%% account for dosing for each new dosing interval
    X0(S.bolus)   = X0(S.bolus) + study.bolus.dose*model.SF.mg_to_nmol/model.V.iv_bolus;
    X0(S.IVbag)   = study.infusion.dose*model.SF.mg_to_nmol; 
    X0(S.GItract) = X0(S.GItract) + study.po.dose*model.SF.mg_to_nmol;

    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] = ode15s(@GenericPBPKmodel_RHS,study.observationTime,X0',[],individual);
    
    %%% transform relative to absolute time and modify last time point
    %%% to allow for trough value measurement
    Tend  = max(study.observationTime);
    t = t + (d-1)*Tend; 
    if d~=study.numberOfDosing
        t(end)  = t(end) - 1e-10; 
    end;
    
    %%% store current output in sim structure
    sim.t = [sim.t; t];
    sim.X = [sim.X; X];

    X0 = X(end,:)'; 

end;

%%% determine standart simulation output
%%%
individual = GenericPBPKmodel_determineStandartOutput(individual,sim);

%%% graphical output specific to this model
%%%
GenericPBPKmodel_specificGraphicalOutput(individual)

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_defineModelParameters(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model   = individual.model;

%%% -----------------------------------------------------------------------
%%% starting point of the lumped PBPK model is the default 
%%% well-stirred vascular&tissue model with 13 compartments
%%% This part is identical to the corresponding part in 
%%% GenericPBPK_sMD_wellstirredTissueModel.m
%%%
%%% -----------------------------------------------------------------------

%%% tissue indexing for detailed PBPK model that is to be lumped
%%%
T = individual.T;

T.allTis      = [T.lun T.art T.adi T.bra T.hea T.kid T.mus T.bon T.ski T.gut T.spl T.liv T.ven];
T.allTisExBlo = [T.lun       T.adi T.bra T.hea T.kid T.mus T.bon T.ski T.gut T.spl T.liv      ];
T.artInflow   = [            T.adi T.bra T.hea T.kid T.mus T.bon T.ski T.gut T.spl            ];
T.intoVen     = [            T.adi T.bra T.hea T.kid T.mus T.bon T.ski             T.liv      ];
T.intoLiv     = [                                                      T.gut T.spl            ];
T.tissueDB    = T.allTis; % specify which tissues are part of the general tissue DB

T.initialize.tis.NaN   = NaN*ones(1,max(T.allTis));


%%% -----------------------------------------------------------------------
%%% Define PBPK parameters 
%%%

a   = T.allTis;
sMD = individual.drug;

%%% tissue volumes 
%%%
V.tis        = T.initialize.tis.NaN;
V.tis(a)     = individual.V.tis(a);
% set V.tis=0 for T.art and T.ven; and account for them with V.vas
V.tis([T.art T.ven]) = 0; 

V.vas        = T.initialize.tis.NaN;
V.vas(a)     = individual.V.vas(a);

V.tot        = V.vas + V.tis;

%%% volume for i.v. bolus dosing
%%%
V.iv_bolus   = V.tot(T.ven);  

%%% total tissue-to-blood partition coefficients
%%%
K.tis_up     = T.initialize.tis.NaN;
K.tis_up(a)  = sMD.K.tis_up(a);

K.tis        = sMD.fuP/sMD.BP*K.tis_up; 
K.tot        = (V.tis.*K.tis + V.vas) ./ V.tot;

%%% intrinsic tissue clearance
%%%
CLint.tis    = T.initialize.tis.NaN;
CLint.tis(a) = sMD.CLint.tis(a);

%%% blood flows (ensure closed circulatory system!)
%%%
Q.blo        = T.initialize.tis.NaN;
Q.blo(a)     = individual.Q.blo(a);
Q.blo([T.ven T.lun T.art]) = sum(Q.blo(T.intoVen));


%%% extraction ratio based on well-stirred tissue model
%%%
E            = sMD.E; % keep E.gut and E.feces
E.tis        = T.initialize.tis.NaN;
E.tis(a)     = 0;
% only liver assumed to be extracting organ
e            = T.liv; 
E.tis(e)     = (CLint.tis(e).*K.tis(e))./(Q.blo(e)+CLint.tis(e).*K.tis(e));

%%% elimination corrected partition coefficients
eK.tis       = (1-E.tis).*K.tis; 
eK.tot       = (V.tis.*eK.tis + V.vas) ./ V.tot;

%%% define hepatic clearance wrt. joint vascular & tissue concentration
%CLint.tot    = CLint.tis.*K.tis./K.tot;

%%% first order po absorption rate constant
lambda_po    = individual.drug.lambda_po;

%%% (oral) bioavailablility 
F.bio = (1-E.tis(T.liv))*(1-E.gut)*(1-E.feces);
F.gh  = (1-E.gut)*(1-E.feces);



%%% -----------------------------------------------------------------------
%%% Determine lumped PBPK model parameter values based on the above 
%%% parameter values of the detailed PBPK model
%%%
%%% -----------------------------------------------------------------------


%%% -----------------------------------------------------------------------
%%% Define indexing of lumped PBPK model
%%%

lumping = model.lumping; 

%%% Check, whether lumping contains exactly the same tissues as the above
%%% details PBPK model
%%%
tissues = [];
for n = 1:length(lumping)
    for tis = lumping{n}
        tissues = union(tissues,tis);
    end;
end
if ~isempty(setdiff(T.allTis,tissues))
    message = 'Tissues listed in >>lumping<< do not match the detailed PBPK model to be lumped!';
    GenericPBPKmodel_ReportErrorMessage(message);
end;

S.maxIndex.tis = length(lumping); 
S.allTis = 1:S.maxIndex.tis;

%%% determine lumped compartments containing ven, liv and kid 
for n=S.allTis
    if ismember(T.ven,lumping{n}), S.Cen = n; end;
    if ismember(T.liv,lumping{n}), S.Liv = n; end;
    if ismember(T.kid,lumping{n}), S.Kid = n; end;
end;
S.allExCen = setdiff(S.allTis,S.Cen);
S.bolus    = S.Cen; % i.v. bolus administration 

S.maxIndex.tis = max(S.allTis);
S.initialize.tis.NaN   = NaN*ones(1,S.maxIndex.tis);

%%% dosing and metabolism indexing

S.GItract = 1 + S.maxIndex.tis; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.tis; % IVbag (for infusion dosing)
S.metab   = 3 + S.maxIndex.tis; % metabolized drug

S.maxIndex.dosing = 3 + S.maxIndex.tis;

%%% additional indexing
%%%

S.maxIndex.all = 0 + S.maxIndex.dosing;

%%% ODE indexing (for relative indexing)
%%%
S.C_lump = 1:S.maxIndex.tis; % all tissues


%%% -----------------------------------------------------------------------
%%% Define lumped PBPK model parameters 
%%%

V.lump  = S.initialize.tis.NaN; 
Q.lump  = S.initialize.tis.NaN;
CL.lump = S.initialize.tis.NaN;
eK.lump = S.initialize.tis.NaN;
for n=S.allTis
    ind = lumping{n};
    V.lump(n)  = sum(V.tot(ind)); 
    Q.lump(n)  = sum(Q.blo(ind));
    CL.lump(n) = sum(Q.blo(ind).*E.tis(ind));
    eK.lump(n) = sum(V.tot(ind).*eK.tot(ind))/V.lump(n);
end;
Q.lump(S.Cen) = sum(Q.lump(S.allExCen));

%%% volume for i.v. bolus dosing
%%%
V.iv_bolus = V.lump(S.Cen);

%%% initial condition of ODE
%%%
X0 = zeros(1,S.maxIndex.all);


%%% -----------------------------------------------------------------------
%%% Assign model parameters 
%%%

model.S     = S;
model.T     = T;
model.V     = V;
model.Q     = Q;
model.K     = K;
model.CL    = CL;
model.E     = E;
model.F     = F;
model.eK    = eK;
model.lambda_po = lambda_po;
model.SF = sMD.SF;
model.X0 = X0;

individual.model = model;
end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = individual.model; S = model.S;

%%% variables (always use row vector notation)
C_lump    = X(S.C_lump)';
A_GItract = X(S.GItract)';

%%% tissue volumes, blood flows, extraction ratios, clearance etc.
V  = model.V.lump;
Q  = model.Q.lump; 
CL = model.CL.lump;
eK = model.eK.lump;
F  = model.F;
infusion = individual.study.infusion;

infusion_rate = 0;
if (infusion.tend > 0) && (t<=infusion.tend)
    infusion_rate = model.SF.mg_to_nmol * infusion.dose/infusion.tend;
end;
lambda_po = model.lambda_po; 

if S.Liv == S.Cen
    F.lump = F.bio;
else
    F.lump = F.gh;
end;

%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
VdC_lump = zeros(size(C_lump));

if length(S.allTis) > 1
    Cin_cen         = sum(Q(S.allExCen).*C_lump(S.allExCen)./eK(S.allExCen)) / Q(S.Cen);
    
    VdC_lump(S.allExCen) = Q(S.allExCen).*(C_lump(S.Cen)/eK(S.Cen) ...
                          -C_lump(S.allExCen)./eK(S.allExCen));
    VdC_lump(S.Cen)      = Q(S.Cen)*(Cin_cen - C_lump(S.Cen)/eK(S.Cen)) ...
                           -CL(S.Liv)*C_lump(S.Liv)/eK(S.Liv) + infusion_rate; 
else
    VdC_lump(S.Cen)      = -CL(S.Liv)*C(S.Liv)/eK(S.Liv) + infusion_rate; 
end;
VdC_lump(S.Liv) = VdC_lump(S.Liv)  + F.lump*lambda_po*A_GItract;

%%% drug amount in GItract for absorption
dA_GItract = -lambda_po*A_GItract;

%%% drug amount in IVbag for infusion
dA_IVbag = -infusion_rate;

%%% metabolized and excreted compound (mass)
dA_metab = CL(S.Liv)*C_lump(S.Liv)/eK(S.Liv) + (1-F.lump)*lambda_po*A_GItract; 

%%%
%%% END OF ODEs 
%%% -----------------------------------------------------------------------


%%% converting amounts to concentrations
dC_lump = zeros(size(C_lump)); 
dC_lump(S.allTis) = VdC_lump(S.allTis)./V(S.allTis);

%%% output vector (always in column vector notation)
dX(S.C_lump)  = dC_lump';
dX(S.GItract) = dA_GItract';
dX(S.IVbag)   = dA_IVbag';
dX(S.metab)   = dA_metab';

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_determineStandartOutput(individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model; info = individual.info; S = model.S; 
pred  = individual.pred; 

%%% assign predicted profiles
%%%
info.pred.t         = 'simulation time in [min]';
pred.t              = sim.t;

info.pred.C.lump    = 'concentration-time profile in lumped total tissue space [nmol/L]';
pred.C.lump         = sim.X(:,S.C_lump);

info.pred.A.tot     = 'amount in total tissue space [nmol]';
pred.A.tot          = pred.C.lump(:,S.allTis)*diag(model.V.lump(S.allTis));

info.pred.A.body    = 'total amount in the body in [nmol]';
pred.A.body         = sum(pred.A.tot,2);

info.pred.A.GItract = 'remaining amount in the GI tract in [nmol]';
pred.A.GItract      = sim.X(:,S.GItract);

info.pred.A.IVbag   = 'remaining amount in the IVbag in [nmol]';
pred.A.IVbag        = sim.X(:,S.IVbag);

info.pred.A.metab   = 'amount metabolized in [nmol]';
pred.A.metab        = sim.X(:,S.metab);

%%% transform lumped concentrations back to original concentrations
%%% initialize all compartments with NaN and assign only to those
%%% compartments values that are part of the model topology

info.pred.C.tot = 'concentration-time profile in total tissue space [nmol/L]';
pred.C.tot = ones(size(pred.t))*individual.stdout.T.initialize.tissueDB.NaN;
for n = S.allTis
    
    nC = pred.C.lump(:,n)/model.eK.lump(n);
    for tis = model.lumping{n}
        pred.C.tot(:,tis) = model.eK.tot(tis)*nC;
    end;
    
end; 

%%% determine standard output in [mg/L] or [mg]
%%%
stdout = individual.stdout; T = stdout.T;
initialize.NaN = ones(size(pred.t))*T.initialize.tissueDB.NaN; SF = model.SF; 

info.stdout.t         = 'simulation time in [min]';
stdout.t              = pred.t;

info.stdout.C.tis     = 'concentration-time profile in tissue space [mg/L]';
stdout.C.tis          = initialize.NaN;
for k = intersect(T.tissueDB,model.T.tissueDB)
   stdout.C.tis(:,k)  = SF.nmol_to_mg*pred.C.tot(:,k)*(model.eK.tis(k)./model.eK.tot(k));
end;
stdout.C.tis(:,T.pla) = stdout.C.tis(:,T.ven)/individual.drug.BP;

info.stdout.nC.tis    = 'normalized concentration-time profile (nC.tis = C.tis/eK.tis) in tissue space [mg/L] ';
stdout.nC.tis         = initialize.NaN;
for k = intersect(T.tissueDB,model.T.tissueDB)
   stdout.nC.tis(:,k) = stdout.C.tis(:,k)/model.eK.tis(k);
end;

info.stdout.A.body    = 'total amount in the body in [mg]';
stdout.A.body         = SF.nmol_to_mg*pred.A.body;

info.stdout.A.GItract = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract      = SF.nmol_to_mg*pred.A.GItract;

info.stdout.A.IVbag   = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag        = SF.nmol_to_mg*pred.A.IVbag;

info.stdout.A.metab   = 'amount metabolized in [mg]';
stdout.A.metab        = SF.nmol_to_mg*pred.A.metab;


individual.info   = info;
individual.pred   = pred;
individual.stdout = stdout;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_specificGraphicalOutput(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% define here specific output for this model

end



%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

