%%% 
%%% This function specifies the virtual population to be studied
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_virtualPopulation(reference,population)

fprintf(' & generating virtual population >>%s<<',population.how2generate);

switch population.how2generate
    
% ======================================================================= %
    case 'human_sMD_PBPK_male'   

        id = 1; individual(id) = reference.human35m; 
        individual(id).id = id; individual(id).color = 'r-'; 
        
% ======================================================================= %
    case 'human_sMD_PBPK_adults' 
        
        id = 1; individual(id) = reference.human35m; 
        individual(id).id = id; individual(id).color = 'r-'; 
        
        id = 2; individual(id) = reference.human35f; 
        individual(id).id = id; individual(id).color = 'b-'; 
            
% ======================================================================= %
    case 'human_sMD_PBPK_all'  
        
        id = 1; individual(id) = reference.human35m; 
        individual(id).id = id; individual(id).color = 'r-'; 
        
        id = 2; individual(id) = reference.human35f; 
        individual(id).id = id; individual(id).color = 'b-'; 

        id = 3; individual(id) = reference.human15m; 
        individual(id).id = id; individual(id).color = 'r--'; 
        
        id = 4; individual(id) = reference.human15f; 
        individual(id).id = id; individual(id).color = 'b--'; 
        
        id = 5; individual(id) = reference.human10; 
        individual(id).id = id; individual(id).color = 'k-'; 
        
        id = 6; individual(id) = reference.human5; 
        individual(id).id = id; individual(id).color = 'k--'; 
        
        
% ======================================================================= %
    case 'rat_sMD_PBPK'

        id = 1; individual(id) = reference.rat250; 
        individual(id).id = id; individual(id).color = 'r-'; 
        
        id = 2; individual(id) = reference.rat475; 
        individual(id).id = id; individual(id).color = 'b-'; 

        
% ======================================================================= %
    case 'rat250_male_sMD_PBPK' 
        
        id = 1; individual(id) = reference.rat250; 
        individual(id).id = id; individual(id).color = 'r-'; 
        
        id = 2; individual(id) = reference.human35m; 
        individual(id).id = id; individual(id).color = 'b-'; 

        
% ======================================================================= %
    case 'rat_human_sMD_PBPK'  
        
        id = 1; individual(id) = reference.human35m; 
        individual(id).id = id; individual(id).color = 'r-'; 
        
        id = 2; individual(id) = reference.human35f; 
        individual(id).id = id; individual(id).color = 'b-'; 

        id = 3; individual(id) = reference.human15m; 
        individual(id).id = id; individual(id).color = 'r--'; 
        
        id = 4; individual(id) = reference.human15f; 
        individual(id).id = id; individual(id).color = 'b--'; 
        
        id = 5; individual(id) = reference.human10; 
        individual(id).id = id; individual(id).color = 'k-'; 
        
        id = 6; individual(id) = reference.human5; 
        individual(id).id = id; individual(id).color = 'k--'; 
        
        id = 7; individual(id) = reference.rat250; 
        individual(id).id = id; individual(id).color = 'm-'; 
        
        id = 8; individual(id) = reference.rat475; 
        individual(id).id = id; individual(id).color = 'm--'; 
        
% ======================================================================= %
    case 'sMD_PBPK_identical' % N = population.size many individual of subtype population.subtype
        
        subtype = population.subtype;
        individual(population.size) = struct(reference.(subtype)); % allocate memory
        
        for id=1:population.size 
               individual(id) = reference.(subtype);
               individual(id).id = id; individual(id).color = {'k'};
        end;
        
% ======================================================================= %
    case 'human_sMD_PBPK_random' % based on de la Grandmaison characteristics
        %%% the original puplication by de la Grandmaison et al (2001)
        %%% contained 355 male/329 female adults 

        
        numberOfMale = population.size; numberOfFemale = 0;
        population = delaGrandmaison(numberOfMale,numberOfFemale);
        individual(population.size) = struct(reference.human35m); % allocate memory
        
        for id=1:population.size 
               
            subtype = char(population.subtype(id));        
            method.BSD = 'LBW'; method.BW = population.BW(id); method.BH = population.BH(id); 
            
            individual(id) = GenericPBPKmodel_scaleIndividual(reference.(subtype),method);         
            individual(id).id = id; individual(id).color = population.color(id);
            
        end;
        
% ======================================================================= %
   case 'human_sMD_PBPK_obese'   % generates 5 individuals with varying BW resp. BMI
       
        
        subtype = population.subtype;
        BW_ref = reference.(subtype).BW; BH_ref = reference.(subtype).BH;
        method.BSD = 'LBW'; method.BH = BH_ref;
        
        id = 1; method.BW = 0.7 * BW_ref;
        individual(id) = GenericPBPKmodel_scaleIndividual(reference.(subtype),method);
        individual(id).id = id; individual(id).color = 'm-'; 

        id = 2; method.BW = 1.0 * BW_ref;
        individual(id) = reference.(subtype);
        individual(id).id = id; individual(id).color = 'r-'; 

        id = 3; method.BW = 1.3 * BW_ref;
        individual(id) = GenericPBPKmodel_scaleIndividual(reference.(subtype),method);
        individual(id).id = id; individual(id).color = 'b--'; 
        
        id = 4; method.BW = 2.0 * BW_ref;
        individual(id) = GenericPBPKmodel_scaleIndividual(reference.(subtype),method);
        individual(id).id = id; individual(id).color = 'b--'; 

        
% ======================================================================= %
    case 'sMD_PBPK_lumping' % compare full to lumped model 
                
        subtype = population.subtype; T = reference.(subtype).T;

        id = 1; individual(id) = reference.(subtype);
        individual(id).id = id; individual(id).color = {'r-'};
        individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';      
        individual(id).color = {'r-'};

        lumping = {[T.ven,T.lun,T.art,T.bra,T.hea,T.kid,T.gut,T.spl,T.liv,T.ski],[T.adi,T.mus,T.bon]};   

        id = 2; individual(id) = reference.(subtype);
        individual(id).id = id; individual(id).color = {'b--'};
        individual(id).model.type = 'sMD_lumpedPBPK_xCMT_wellstirred'; 
        individual(id).color = {'b--'};
        individual(id).model.lumping = lumping;



% ======================================================================= %
    case 'sMD_PBPK_vs_CMT' % compare PBPK to classical compartment model 
                           % with C_central = C_plasma (not C_blood)!
        
        subtype = population.subtype;

        id = 1; individual(id) = reference.(subtype);
        individual(id).id = id; individual(id).color = {'r-'};
        individual(id).model.type = 'sMD_PBPK_13CMT_wellstirred';            
        
        id = 2; individual(id) = reference.(subtype);
        individual(id).id = id; individual(id).color = {'b--'};
        individual(id).model.type = 'CMTmodel_xCMT_PLASMA_macroConstants_linearCL';
        individual(id).model.V         = [12.8*2.4*0.85 54.3*2.4*0.85];  % in [L]
        individual(id).model.Q         = [1.76*0.85];    % in [L/min]
        individual(id).model.CL        = [1.095*0.85 0]; % in [L/min]
        individual(id).model.F.bio     = 1.0;       % set to NaN if not needed
        individual(id).model.lambda_po = 0.018;     % set to NaN if not needed
        
        
% ======================================================================= %
    case 'mouse_mAb_PBPK'
        
        mouse = reference.mouseFaH;
        
        id = 1; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'b-'};
        individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
        individual(id).model.correct_for_residual_blood = 1;
        individual(id).model.FcRn_status = 'wild-type';
        
        id = 2; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'b--'};
        individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
        individual(id).model.correct_for_residual_blood = 0;
        individual(id).model.FcRn_status = 'wild-type';
        
        id = 3; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'k-'};
        individual(id).model.type = 'ExperimentalData';
        individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';
        
        showAlsoKnockout = 0;
        if showAlsoKnockout
            id = 4; individual(id) = mouse;
            individual(id).id = id; individual(id).color = {'r-'};
            individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
            individual(id).model.correct_for_residual_blood = 1;
            individual(id).model.FcRn_status = 'knock-out';
            
            id = 5; individual(id) = mouse;
            individual(id).id = id; individual(id).color = {'r--'};
            individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
            individual(id).model.correct_for_residual_blood = 0;
            individual(id).model.FcRn_status = 'knock-out';
            
            id = 6; individual(id) = mouse;
            individual(id).id = id; individual(id).color = {'k-'};
            individual(id).model.type = 'ExperimentalData';
            individual(id).model.filename = 'ExperimentalData_mouse_ko_mAb7E3_Garg2007';
        end;
        

% ======================================================================= %
    case 'mouse_mAb_PBPK_ABC'
        
        mouse = reference.mouseFaH;
        
        id = 1; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'r-'};
        individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
        individual(id).model.useABCs = 'no'; % 'yes' or 'no' = anything else
        individual(id).model.correct_for_residual_blood = 1;
        individual(id).model.FcRn_status = 'wild-type';
        
        id = 2; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'b--'};
        individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
        individual(id).model.useABCs = 'yes'; % 'yes' or 'no' = anything else
        individual(id).model.correct_for_residual_blood = 0;
        individual(id).model.FcRn_status = 'wild-type';
        
        id = 3; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'k-'};
        individual(id).model.type = 'ExperimentalData';
        individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';
        
% ======================================================================= %
    case 'mouse_mAb_PBPK_vs_HaB' 
        
        mouse = reference.mouseFaH;
        
        id = 1; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'b--'};
        individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
        individual(id).model.FcRn_status = 'wild-type';
        individual(id).model.correct_for_residual_blood = 1;
        
        id = 2; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'r-'};
        individual(id).model.type = 'mAb7E3_mouse_2CMT_corrHansenBalthasar';
        
        id = 3; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'k-'};
        individual(id).model.type = 'ExperimentalData';
        individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';

        
% ======================================================================= %
    case 'mouse_mAb_PBPK_lumping' 
        
        mouse = reference.mouseFaH; T = mouse.T;

        lumping = {[T.adi,T.bon,T.mus,T.ski],[T.liv,T.gut,T.hea,T.kid,T.spl,T.lun],[T.pla]};

        id = 1; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'b-'};
        individual(id).model.type = 'mAb_lumpedPBPK_xCMT_extravasationLimited';
        individual(id).model.FcRn_status = 'wild-type';
        individual(id).model.scenario = 7;
        individual(id).model.useABCs = 'no'; % 'yes' or 'no' = anything else
        individual(id).model.correct_for_residual_blood = 1;
        individual(id).model.lumping = lumping;
        
        id = 2; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'r--'};
        individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
        individual(id).model.FcRn_status = 'wild-type';
        individual(id).model.scenario = 7;
        individual(id).model.useABCs = 'no'; % 'yes' or 'no' = anything else
        individual(id).model.correct_for_residual_blood = 1;

        id = 3; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'k-'};
        individual(id).model.type = 'ExperimentalData';
        individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';

   
% ======================================================================= %
    case 'mouse_mAb_PBPK_target'
        
        mouse = reference.mouseFaH;
        
        id = 1; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'r-'};
        individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited';
        individual(id).model.correct_for_residual_blood = 0;
        individual(id).model.scenario = 7;
        individual(id).model.FcRn_status = 'wild-type';
        
        id = 2; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'b--'};
        individual(id).model.type = 'mAb_PBPK_11CMT_extravasationLimited_Target';
        individual(id).model.FcRn_status = 'wild-type';
        individual(id).model.scenario = 7;
        individual(id).model.useABCs = 'no'; % 'yes' or 'no' = anything else
        individual(id).model.correct_for_residual_blood = 0;        
        % tissues                    lun adi hea kid mus bon ski gut spl liv pla
        individual(id).model.V_max = [0   3   0   0   0   0   3   0   0   3   0 ]*1e-6; 
        individual(id).model.B_max = [0   6   0   0   0   0   6   0   0   6   0 ]*1e1; 
        individual(id).model.K_m   = [1   1   1   1   1   1   1   1   1   1   1 ]*0.12; 

        id = 3; individual(id) = mouse;
        individual(id).id = id; individual(id).color = {'k-'};
        individual(id).model.type = 'ExperimentalData';
        individual(id).model.filename = 'ExperimentalData_mouse_wt_mAb7E3_Garg2007';

        
% ======================================================================= %
    otherwise
        
        message = sprintf('Unknow approach >>%s<< of generating a virtual population!',population.how2generate);
        GenericPBPKmodel_ReportErrorMessage(message);
        
end; % of switch population.how2generate

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function population = delaGrandmaison(numberOfMales,numberOfFemales)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


population.size = numberOfMales + numberOfFemales;
% male
idVec = 1:numberOfMales;
for id=idVec
    population.subtype(id) = {'human35m'};
    population.color(id)   = {'k'};
end;
muBH  = 1.72;
sdBH  = 0.075;
muBMI = 22.8;
sdBMI = 3.3;
population.BH(idVec) = muBH + sdBH * randn(1,numberOfMales);

% transforming mu and sd to the scale of the underlying normal distribution
log_muBMI = log(muBMI^2/sqrt(muBMI^2+sdBMI^2));
log_sdBMI = sqrt(log(sdBMI^2/muBMI^2+1));

% BMI lognormal with mean=muBMI and sd=s
population.BMI(idVec) = exp(log_muBMI + log_sdBMI * randn(1,numberOfMales));
population.BW(idVec)  = population.BMI(idVec) .* (population.BH(idVec).^2);

% female
idVec = [1:numberOfFemales]+numberOfMales;
for id=idVec
    population.subtype(id) = {'human35f'};
    population.color(id)   = {'b'};
end;
muBH  = 1.61;
sdBH  = 0.075;
muBMI = 22.5;
sdBMI = 4.5;
population.BH(idVec) = muBH + sdBH * randn(1,numberOfFemales);

% transforming mu and sd to the scale of the underlying normal distribution
log_muBMI = log(muBMI^2/sqrt(muBMI^2+sdBMI^2));
log_sdBMI = sqrt(log(sdBMI^2/muBMI^2+1));

% BMI lognormal with mean=muBMI and sd=s
population.BMI(idVec) = exp(log_muBMI + log_sdBMI * randn(1,numberOfFemales));
population.BW(idVec)  = population.BMI(idVec) .* (population.BH(idVec).^2);

end



%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

