%%% 
%%% This function specifies the corrected Hansen and Balthasr model 
%%% published in J Pharmacokinet Pharmacodyn (2003)
%%% 
%%% Version: February 10th, 2014. 
%%% For references and citation, please see MAIN script.
%%% 
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the 
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported 
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function individual = GenericPBPKmodel_mAb7E3_mouse_2CMT_corrHansenBalthasar(individual)

%%% Corrected Hansen and Balthasar model. The correction is w.r.t. the 
%%% FcRn interaction term (using peripheral instead of central concentration).
%%% The output is blood concentration of endogenous IgG and the mAb (7E3)

%%% define model specific parameters
%%%
individual = GenericPBPKmodel_defineModelParameters(individual);

%%% simulate classical CMT model 
%%%
model = individual.model; S = model(1).S; study = individual.study(1); SF = model(1).SF;
X0 = model.X0; sim.t = []; sim.X = []; 

for d=1:study.numberOfDosing
    
    %%% account for dosing for each new dosing interval
    X0(S.bolus)   = X0(S.bolus) + study.bolus.dose*SF.mg_to_nmol/model(1).V.iv_bolus;
    X0(S.IVbag)   = study.infusion.dose*SF.mg_to_nmol; 

    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] = ode15s(@GenericPBPKmodel_RHS,study.observationTime,X0',[],individual);
    
    %%% transform relative to absolute time and modify last time point
    %%% to allow for trough value measurement
    Tend  = max(study.observationTime);
    t = t + (d-1)*Tend; 
    if d~=study.numberOfDosing
        t(end)  = t(end) - 1e-10; 
    end;
    
    %%% store current output in sim structure
    sim.t = [sim.t; t];
    sim.X = [sim.X; X];

    X0 = X(end,:)'; 

end;

%%% determine standart simulation output
%%%
individual = GenericPBPKmodel_determineStandartOutput(individual,sim);

%%% make specific plots
%%%
GenericPBPKmodel_specificGraphicalOutput(individual);

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_defineModelParameters(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

mAb = individual.drug;

%%% tissue indexing
%%%
S = individual.T;

S.cen = 1; S.per = 2;
S.allCmt       = [S.cen S.per];
S.maxIndex.cmt = max(S.allCmt);
S.bolus        = S.cen;

S.initialize.cmt.NaN   = NaN*ones(1,max(S.allCmt));

%%% dosing and metabolism indexing

S.GItract = 1 + S.maxIndex.cmt; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.cmt; % IVbag (for infusion dosing)
S.metab   = 3 + S.maxIndex.cmt; % metabolized drug

S.maxIndex.dosing = 3 + S.maxIndex.cmt;

%%% additional indexing for IgG (same cmt indexing as for mAb)
%%%

S.maxIndex.all = S.maxIndex.cmt + S.maxIndex.dosing;

%%% ODE indexing (for relative indexing)
%%%
S.C_mAb =  1:S.maxIndex.cmt; 
S.C_IgG = (1:S.maxIndex.cmt) + S.maxIndex.dosing;



%%% -----------------------------------------------------------------------
%%% Define CMT model parameters 

SF = mAb.SF;
SF.day_to_min = 1/(24*60);  % to scale parameters form day --> min

%%% molecular weight, total FcRn concentration and FcRn-affinity 
%%% (assumed to be identical for endogenous IgG and 7E3)
MW        = 150*1e3;     % [g/mol]
FcRn_tot  = 2.219e+5;    % [nmol/L]
KD        = 4.8;         % [nmol/L]

%%% volumes
%%%
V.cmt = S.initialize.cmt.NaN;
V.cmt(S.allCmt) = [1.6 0.088]/1000;    %[mL-> L]

%%% volume for i.v. bolus dose
V.iv_bolus = V.cmt(S.cen);

%%% Intercompartmental clearances 
Q.in    = 1.597 *SF.day_to_min/1000;    %[mL/d]->[L/min]
Q.out   = 0.102 *SF.day_to_min/1000;    %[mL/d]->[L/min]
CL      = 0.043 *SF.day_to_min/1000;    %[mL/d]->[L/min]

%%% first order oral absorption rate constant in [1/min]

if individual.study.po.dose>0
    GenericPBPKmodel_ReportErrorMessage('po administration of mAb not supported!');
end;

%%% endogenous IgG synthesis rate
%%%
r_synth = 1.849*SF.day_to_min;    %[nmol/d] -> [nmol/min]

%%% initial condition of ODE
%%%
X0 = zeros(1,S.maxIndex.all);
X0(S.C_IgG) = [0.2294e10 3.9715e10]/MW; % [nmol/L] steady state endogenous IgG concentration

%%% -----------------------------------------------------------------------
%%% Assign model parameters 
%%% 
model.S          = S;  
model.SF         = SF;
model.MW         = MW;
model.FcRn_tot   = FcRn_tot;
model.KD         = KD;
model.V          = V;
model.Q          = Q;
model.CL         = CL;
model.r_synth    = r_synth;
model.X0         = X0;


individual.model = model;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = individual.model; S = model.S;

%%% variables (always use row vector notation)
C_mAb = X(S.C_mAb)';
C_IgG = X(S.C_IgG)';

%%% common parameter values for tissue volumes, blood flows, clearance etc.
V  = model.V.cmt;
Q  = model.Q;
CL = model.CL;
MW         = model.MW;
FcRn_tot   = model.FcRn_tot;
KD         = model.KD;

%%% specific parameter values for (i) mAb
%%%
infusion = individual.study.infusion;

infusion_rate = 0;
if (infusion.tend > 0) && (t<=infusion.tend)
    infusion_rate = model.SF.mg_to_nmol * infusion.dose/infusion.tend;
end;

%%% specific parameter values for (ii) endogeneous IgG
%%%
r_synth = model.r_synth;

%%% total mAb and endog. IgG concentration; [mg/L]->[g/L]->[mol/L]=[M]
%%%
C_tot    = C_mAb(S.per) + C_IgG(S.per); 
FcRn_eff = FcRn_tot-C_tot-KD;
FcRn_u   = 1/2*( FcRn_eff + sqrt(FcRn_eff^2 + 4*KD.*FcRn_tot) );

%%% Fraction unbound
%%%
fu = KD/(KD+FcRn_u);


%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
VdC_mAb = zeros(size(C_mAb)); VdC_IgG = zeros(size(C_IgG)); 

VdC_mAb(S.cen) = -Q.in*C_mAb(S.cen) + Q.out*(1-fu)*C_mAb(S.per) ...
                 +infusion_rate;
VdC_mAb(S.per) =  Q.in*C_mAb(S.cen) - Q.out*(1-fu)*C_mAb(S.per) ...
                 -CL*fu*C_mAb(S.per);

VdC_IgG(S.cen) = -Q.in*C_IgG(S.cen) + Q.out*(1-fu)*C_IgG(S.per) ...
                 +r_synth;
VdC_IgG(S.per) =  Q.in*C_IgG(S.cen) - Q.out*(1-fu)*C_IgG(S.per) ...
                 -CL*fu*C_IgG(S.per);

%%% mAb amount in IVbag for infusion
dA_IVbag = -infusion_rate;

%%% mAb amount metabolized or excreted
dA_metab = +CL*fu*C_mAb(S.per);

%%%
%%% END OF ODEs 
%%% -----------------------------------------------------------------------

%%% converting amounts to concentrations
dC_mAb = zeros(size(C_mAb)); 
dC_mAb(S.allCmt) = VdC_mAb(S.allCmt)./V(S.allCmt);

dC_IgG = zeros(size(C_IgG)); 
dC_IgG(S.allCmt) = VdC_IgG(S.allCmt)./V(S.allCmt);


%%% output vector (always in column vector notation)
dX(S.C_mAb) = dC_mAb';
dX(S.C_IgG) = dC_IgG';
dX(S.IVbag) = dA_IVbag';
dX(S.metab) = dA_metab';

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function individual = GenericPBPKmodel_determineStandartOutput(individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = individual.model; info = individual.info; S = model.S; 

%%% assign predicted profiles for (i) mAb
%%%
info.pred.t         = 'simulation time in [min]';
pred.t              = sim.t;

info.pred.C.mAb     = 'concentration-time profile of mAb in different compartents [nmol/L]';
pred.C.mAb          = sim.X(:,S.C_mAb);

info.pred.A.tot     = 'amount of mAb in different compartments [nmol]';
pred.A.tot          = pred.C.mAb(:,S.allCmt)*diag(model.V.cmt(S.allCmt));

info.pred.A.body    = 'total amount of mAb in the body in [nmol]';
pred.A.body         = sum(pred.A.tot,2);

info.pred.A.GItract = 'remaining amount of mAb in the GI tract in [nmol]';
pred.A.GItract      = sim.X(:,S.GItract);

info.pred.A.IVbag   = 'remaining amount of mAb in the IVbag in [nmol]';
pred.A.IVbag        = sim.X(:,S.IVbag);

info.pred.A.metab   = 'amount of mAb metabolized in [nmol]';
pred.A.metab        = sim.X(:,S.metab);

info.pred.C.IgG     = 'concentration-time profile of IgG in different compartents [nmol/L]';
pred.C.IgG          = sim.X(:,S.C_IgG);


%%% determine standard output in [mg/L] or [mg]
%%%
stdout = individual.stdout; T = stdout.T;
initialize.NaN = ones(size(pred.t))*T.initialize.tissueDB.NaN; SF = model.SF; 

info.stdout.t          = 'simulation time in [min]';
stdout.t               = pred.t;

info.stdout.C.tis      = 'concentration-time profile in tissue space [mg/L]';
stdout.C.tis           = initialize.NaN;
stdout.C.tis(:,T.pla)  = SF.nmol_to_mg*pred.C.mAb(:,S.cen);

info.stdout.nC.tis     = 'normalized concentration-time profile (nC.tis = C.tis/eK.tis) in tissue space [mg/L] ';
stdout.nC.tis          = initialize.NaN;
stdout.nC.tis(:,T.pla) = stdout.C.tis(:,T.pla);


info.stdout.A.body     = 'total amount in the body in [mg]';
stdout.A.body          = SF.nmol_to_mg*pred.A.body;

info.stdout.A.GItract  = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract       = SF.nmol_to_mg*pred.A.GItract;

info.stdout.A.IVbag    = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag         = SF.nmol_to_mg*pred.A.IVbag;

info.stdout.A.metab    = 'amount metabolized in [mg]';
stdout.A.metab         = SF.nmol_to_mg*pred.A.metab;


individual.info   = info;
individual.pred   = pred;
individual.stdout = stdout;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_specificGraphicalOutput(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

return;


pred  = individual.pred; S = individual.model(1).S;
t     = pred.t/(60*24); % convert from min to days
C_mAb = pred.C.cmt(:,S.allCmt);
C_IgG = pred.C.cmt(:,S.allCmt);

%%% endogenous IgG in the central and peripheral compartment
%%%

figure(70); clf;
semilogy(t,C_IgG(:,S.cen),'b-',t,C_IgG(:,S.per),'b--');
xlabel('t[days]'); ylabel('endogenous IgG: cen(-), per(--)');

model    = individual.model;
MW       = model(1).MW;
FcRn_tot = model(1).FcRn_tot;
KD       = model(1).KD;

%%% total mAb and endog. IgG concentration; [mg/L]->[g/L]->[mol/L]=[M]
%%%
C_tot    = 1e-3*( C_mAb(:,S.per)/MW + C_IgG(:,S.per)/MW ); 
FcRn_eff = FcRn_tot-C_tot-KD;
FcRn_u   = 1/2*( FcRn_eff + sqrt(FcRn_eff.^2 + 4*KD.*FcRn_tot) );

%%% Fraction unbound
%%%
fu = KD./(KD+FcRn_u);

figure(71); clf;
plot(t,fu/fu(1),'b-');
xlabel('t[days]'); ylabel('change in fu');

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

