%%% Version: October 31st, 2014
%%%
%%% Updated:
%%% (a) introduction of experimental parameter values and subsequent
%%%     explicit statement of their interpretation, e.g., whether
%%%     experimental OW is interpreted as total tissue or tissue weight
%%% (b) amendment of Brown et al data for mice (blood flow data)
%%% (c) renaming of parameters related to blood, e.g, introduction of
%%%     regional peripheral vascular blood as fraction of total blood
%%%
%%% This function specifies anatomical and physiological parameter values
%%% of different species
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%

function species = GenericPBPKmodel_speciesDatabase


%%% ========================================================================================================
%%% CONTENT:
%%%
%%% 1) Data fields, units, description of parameters, tissue indexing
%%% 2) Rat
%%% 3) Mouse
%%% 4) Human
%%%
%%% ========================================================================================================


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: DATA FIELDS, UNITS, PARAMETER DESCRIPTION, TISSUE INDEXING ++++++++++++++++++++++++++++++++++++++
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% ========================================================================================================
%%% == DATA FIELDS, UNITS, and PARAMETER DESCRIPTION
%%%

data.subtype.type       = [];    info.type       = 'species type: rat, mouse, human';
data.subtype.info       = [];    info.info       = 'information on parameters';
data.subtype.T          = [];    info.T          = 'tissue indexing';

% age, sex, (lean) body weight, BMI, BSA
data.subtype.age        = [];    info.age        = 'age of subtype in weeks(animals); years(human)';
data.subtype.sex        = [];    info.sex        = 'sex of subtype: m(ale)/f(emale)/u(nisex)';
data.subtype.BH         = [];    info.BH         = 'body height in [m]';
data.subtype.BW         = [];    info.BW         = 'body weight in [kg]';
data.subtype.LBW        = [];    info.LBW        = 'lean body weight in [kg]';
data.subtype.BMI        = [];    info.BMI        = 'body mass index in [kg/m^2]';
data.subtype.BSA        = [];    info.BSA        = 'body surface area in [m^2]';

% organ weight/fractions, tissue volumes and blood fractions
data.subtype.fBW.OWexp  = [];    info.fBW.OWexp  = 'organ weights as fraction of BW';
data.subtype.fVblood.reg= [];    info.fVblood.reg= 'fraction of blood that is regional vascular blood associated with a tissue';
data.subtype.hct        = [];    info.hct        = 'hematocrit value as fraction of red blood cells-to-blood volume fraction';
data.subtype.dens       = [];    info.dens       = 'tissue density in [kg/L]';
data.subtype.OW.exp     = [];    info.OW.exp     = 'experimental organ weights in [kg]';
data.subtype.OW.tis     = [];    info.OW.tis     = 'tissue organ weights in [kg]';
data.subtype.OW.blood   = [];    info.OW.blood   = 'blood weights in [kg]';

% tissue volumes, constituent fractions and ratios (for partition coefficients)
data.subtype.V          = [];
data.subtype.V.tis      = [];    info.V.tis      = 'tissue volumes (NOT INCLUDING VASCULAR SPACE) in [L]';
data.subtype.V.vas      = [];    info.V.vas      = 'perihperal vascular blood in [L]';
data.subtype.V.tot      = [];    info.V.tot      = 'total tissue volumes (INCLUDING regional vascular blood) in [L]';
data.subtype.fVtis      = [];
data.subtype.fVtis.int  = [];    info.fVtis.int  = 'interstitial volume as fraction of tissue volume';
data.subtype.fVtis.cel  = [];    info.fVtis.cel  = 'intra-cellular volume as fraction of tissue volume';
data.subtype.fVtis.wex  = [];    info.fVtis.wex  = 'extra-cellular water volume as fraction of tissue volume';
data.subtype.fVtis.wic  = [];    info.fVtis.wic  = 'intra-cellular water volume as fraction of tissue volume';
data.subtype.fVtis.nlt  = [];    info.fVtis.nlt  = 'neutral lipid volume as fraction of tissue volume';
data.subtype.fVtis.npt  = [];    info.fVtis.npt  = 'neutral phospholipd volume as fraction of tissue volume';
data.subtype.fVtis.APmt = [];    info.fVtis.APmt = 'intra-cellular acidic phospholipid volume as fraction of tissue volume';
data.subtype.r          = [];
data.subtype.r.Atp      = [];    info.r.Atp      = 'ratio of albumin tissue-to-plasma concentrations';
data.subtype.r.Ltp      = [];    info.r.Ltp      = 'ratio of lipoprotein tissue-to-plasma concentrations';

% cardiac output and blood flows/fractions
data.subtype.co         = [];    info.co         = 'cardiac output in [L/min]';
data.subtype.fQco       = [];    info.fQco       = 'peripheral blood flows as fraction of cardiac output';
data.subtype.Q.blo      = [];    info.Q.blo      = 'peripheral blood flows in [L/min]';


%%% ========================================================================================================
%%% Indexing of organs, tissues and other spaces
%%%

T = GenericPBPKmodel_tissueIndexing;

for  k = T.tissueDB
    
    tis = T.name{k};
    eval(sprintf('%s = %d;',tis,T.(tis)));  %%% this allows to use, e.g., 'lun' instead of 'T.lun'
    
end;


%%% Rest of body weight as well as lun, art & ven blood flows
%%% are determined based on the following reference model
%%%
refModel.T = T;
refModel.T.allTis      = [lun art bra adi hea kid mus bon ski gut spl liv ven];
refModel.T.allTisExBlo = [lun     bra adi hea kid mus bon ski gut spl liv    ];
refModel.T.artInflow   = [        bra adi hea kid mus bon ski gut spl        ];
refModel.T.intoLiv     = [                                    gut spl        ];
refModel.T.intoVen     = [        bra adi hea kid mus bon ski         liv    ];
refModel.T.elimTis     = [                                            liv    ];

%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: DATA FIELDS, UNITS, PARAMETER DESCRIPTION, TISSUE INDEXING ++++++++++++++++++++++++++++++++++++++++
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++++
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% ========================================================================================================
%%% Data: Mostly reported for F344 rats
%%%
%%% Source: Brown et al, Tox Ind Health 1997 (Table 3)
%%%
%%% Note: Male rats of 250g are only 10 weeks of age and in rapid growth phase.
%%% Growth is much slower between age 20-55 weeks (ca. 380-470g) and reaches
%%% a plateau of 475g at age 56-91 weeks.

% rat250 = 1; rat475 = 2;

rat.type    = 'rat';
rat.subtype = {'rat250','rat475'};


%%% ========================================================================================================
%%% Data  : Body weight (BW)
%%%
%%% Units : BW in kg,
%%% Source: Brown et al, Tox Ind Health 1997 (relation between weight and age, see Table 3)
%%%
%%% Strain: most data were reported for the F344 strain

data.rat.sex =   { 'm',  'm'  };
data.rat.age =   [  10    70  ]; % in weeks; 70w = between 56-91w
data.rat.BW  =   [ 0.250 0.475]; %

%%% Assign values to all age/sex classes
for a = 1:length(rat.subtype)
    
    subtype = rat.subtype{a};
    rat.(subtype)         = struct(data.subtype);
    rat.(subtype).type    = 'rat';
    rat.(subtype).info    = info;
    rat.(subtype).T       = T;
    rat.(subtype).age     = data.rat.age(a);
    rat.(subtype).sex     = data.rat.sex{a};
    rat.(subtype).BW      = data.rat.BW(a);
    rat.(subtype).BH      = NaN;
    rat.(subtype).LBW     = NaN;
    rat.(subtype).BMI     = NaN;
    rat.(subtype).BSA     = NaN;
    
end;


%%% ========================================================================================================
%%% Data  : Density of tissue
%%%
%%% Unit  : kg/l (= g/cm^3)
%%%

%%% ASSUMPTION:
%%%
%%% Density assumed identical to human density data
%%% Source: Human data. See Brown et al, Tox Ind Health 1997 and ICRP report 1975
%%%
data.rat.dens      = ones(size(T.initialize.tissueDB.NaN)); % all organs except for adi and bon (see Brown et al, Tox Ind Health 1997, Table 19)
data.rat.dens(adi) = 0.916;                  % ICRP report 1975, p.44
data.rat.dens(bon) = 1.3;                    % ICRP report 2002, Table 2.20 (whole skeleton, adults)


%%% ========================================================================================================
%%% Data  : fraction of total body weight that is experimental organ weight or
%%%         total blood volume
%%%         According to Brown et al. p.411, in most cases, the values provided
%%%         reflect the weight of organs that are drained of blood.
%%%
%%% Unit  : fraction (converted from percentage by dividing by 100]
%%% Source: Brown et al, Tox Ind Health 1997: Table 5 (most tissues),
%%%         Table 13 (adi), top paragraph on p.425 (bone), Table 21 (blo)
%%%

%%%          age(weeks)    10w   70w
data.rat.fBW.OWexp.adi = [10.6  16.0 ]/100;
data.rat.fBW.OWexp.bon = [ 7.3   7.3 ]/100;
data.rat.fBW.OWexp.bra = [ 0.57  0.57]/100;
data.rat.fBW.OWexp.gut = [ 2.7   2.7 ]/100; % sum of stomach, small and large intestine (2.7=0.46+1.40+0.84)
data.rat.fBW.OWexp.hea = [ 0.33  0.33]/100;
data.rat.fBW.OWexp.kid = [ 0.73  0.73]/100;
data.rat.fBW.OWexp.liv = [ 3.66  3.66]/100;
data.rat.fBW.OWexp.lun = [ 0.50  0.50]/100;
data.rat.fBW.OWexp.mus = [40.43 40.43]/100;
data.rat.fBW.OWexp.ski = [19.03 19.03]/100;
data.rat.fBW.OWexp.spl = [ 0.2   0.2 ]/100;

%data.rat.fBW.blood     = [ 7.4   7.4 ]/100; %Brown et al


fprintf('change of rat Vblood\n');

data.rat.fBW.blood     = [ 6.4   6.4 ]/100; %Diehl et al


%%% assign fractional organ weights and tissue volumes
for a = 1:length(rat.subtype)
    
    subtype = rat.subtype{a};
    
    rat.(subtype).fBW.OWexp      = T.initialize.tissueDB.NaN;
    rat.(subtype).fBW.OWexp(adi) = data.rat.fBW.OWexp.adi(a);
    rat.(subtype).fBW.OWexp(bon) = data.rat.fBW.OWexp.bon(a);
    rat.(subtype).fBW.OWexp(bra) = data.rat.fBW.OWexp.bra(a);
    rat.(subtype).fBW.OWexp(gut) = data.rat.fBW.OWexp.gut(a);
    rat.(subtype).fBW.OWexp(hea) = data.rat.fBW.OWexp.hea(a);
    rat.(subtype).fBW.OWexp(kid) = data.rat.fBW.OWexp.kid(a);
    rat.(subtype).fBW.OWexp(liv) = data.rat.fBW.OWexp.liv(a);
    rat.(subtype).fBW.OWexp(lun) = data.rat.fBW.OWexp.lun(a);
    rat.(subtype).fBW.OWexp(mus) = data.rat.fBW.OWexp.mus(a);
    rat.(subtype).fBW.OWexp(ski) = data.rat.fBW.OWexp.ski(a);
    rat.(subtype).fBW.OWexp(spl) = data.rat.fBW.OWexp.spl(a);
    
    %%% fraction of total blood volume, including peripheral regional blood
    rat.(subtype).fBW.blood      = data.rat.fBW.blood(a);
    
    %%% experimental organ weights
    rat.(subtype).OW.exp   = rat.(subtype).fBW.OWexp * rat.(subtype).BW;
    rat.(subtype).OW.blood = rat.(subtype).fBW.blood * rat.(subtype).BW;
    
    %%% ASSUMPTION:
    %%%
    %%% We assume that the experimental organ weights (including residual
    %%% blood to some varying degree) are approximately equal to
    %%% the tissue organ weight (not including residual blood), since
    %%% according to Brown et al. p.411, in most cases, the values provided
    %%% reflect the weight of organs that are drained of blood.
    %%%
    rat.(subtype).OW.tis = rat.(subtype).OW.exp;
    rat.(subtype).LBW    = rat.(subtype).BW  - rat.(subtype).OW.tis(adi);
    
    rat.(subtype).dens   = data.rat.dens;
    rat.(subtype).V.tis  = rat.(subtype).OW.tis ./ rat.(subtype).dens;
    
    %%% total blood volume (assuming a density of 1)
    %%%
    rat.(subtype).V.blood = rat.(subtype).OW.blood;
    
end;


%%% ========================================================================================================
%%% Data  : Fraction of experimental organ weight that is vascular volume and
%%%         interstitial space volume in non-bled rats
%%%
%%% Unit  : fraction in [L/kg]
%%% Source: Kawai et al, J Pharmacokinet Biopharm, Vol 22, 1994 (Table B-I),
%%%         based on measurements in nonbled rats, see Appendix B, p. 362
%%% Note  : The vascular volume fractions in Kawai et al are in good agreement
%%%         with the mean residual blood data in Brown et al, Table 30 (rats)
%%%         There, however, it is mentioned (on p.457, Section 'Blood
%%%         volume data') that the values in Table 30 are not
%%%         representations of the fraction of the total blood volume that
%%%         resides in the tissue.
%%%
data.rat.fOWexp.vas = T.initialize.tissueDB.NaN;
data.rat.fOWexp.vas(adi) = 0.010; % n.a.; Brown etal 1997(Table 30)
data.rat.fOWexp.vas(bon) = 0.041; % 0.04; Brown etal 1997(Table 30)
data.rat.fOWexp.vas(bra) = 0.037; % 0.03; Brown etal 1997(Table 30) = [0.02-0.04], n=3
data.rat.fOWexp.vas(gut) = 0.024; % n.a.; Brown etal 1997(Table 30)
data.rat.fOWexp.vas(hea) = 0.262; % 0.26; Brown etal 1997(Table 30)
data.rat.fOWexp.vas(kid) = 0.105; % 0.16; Brown etal 1997(Table 30) = [0.11-0.27], n=3
data.rat.fOWexp.vas(liv) = 0.115; % 0.21; Brown etal 1997(Table 30) = [0.12-0.27], n=3
data.rat.fOWexp.vas(lun) = 0.262; % 0.36; Brown etal 1997(Table 30) = [0.26-0.52], n=3
data.rat.fOWexp.vas(mus) = 0.026; % 0.04; Brown etal 1997(Table 30) = [0.01-0.04], n=3
data.rat.fOWexp.vas(ski) = 0.019; % 0.02; Brown etal 1997(Table 30) = 0.02
data.rat.fOWexp.vas(spl) = 0.282; % 0.22; Brown etal 1997(Table 30) = [0.17-0.28], n=3

data.rat.fOWexp.int = T.initialize.tissueDB.NaN;
data.rat.fOWexp.int(adi) = 0.135;
data.rat.fOWexp.int(bon) = 0.100;
data.rat.fOWexp.int(bra) = 0.004;
data.rat.fOWexp.int(gut) = 0.094;
data.rat.fOWexp.int(hea) = 0.100;
data.rat.fOWexp.int(kid) = 0.200;
data.rat.fOWexp.int(liv) = 0.163;
data.rat.fOWexp.int(lun) = 0.188;
data.rat.fOWexp.int(mus) = 0.120;
data.rat.fOWexp.int(ski) = 0.302;
data.rat.fOWexp.int(spl) = 0.150;

%%%
data.rat.fOWexp.cel = 1 - (data.rat.fOWexp.vas + data.rat.fOWexp.int);

%%% ASSUMPTION:
%%%
%%% We assume that fractions with respect to total organ weight are
%%% approximately identical to the fractions with respect to experimental
%%% organ weight, since fractions were determined from non-bled rats
%%% according to the footnote of table B-I, Kawai et al. (2004).
%%%
data.rat.fOWtot = data.rat.fOWexp;


%%% ASSUMPTION:
%%%
%%% Moreover, we assume that fractions with respect to volume are identical
%%% to those with respect to weight.
%%%
data.rat.fVtot = data.rat.fOWtot;

%%% Determining fraction of interstitial and intra-cellular space with
%%% respect to tissue weight NOT INCLUDING regional vascular blood so that
%%% fVtis.int+fVtis.cel = 1
%%%
data.rat.fVtis.int = data.rat.fVtot.int./(data.rat.fVtot.int + data.rat.fVtot.cel);
data.rat.fVtis.cel = 1 - data.rat.fVtis.int;


%%% ========================================================================================================
%%% Data  : Fraction of experimental organ weight that is residual blood
%%%         in exanguinated rates
%%%
%%% Unit  : fraction in [L/kg]
%%% Source: Kawai etal, J Pharmacokinet Biopharm, Vol 22, 1994 (Table B-I)
%%%
data.rat.fOWexp.res = T.initialize.tissueDB.NaN; % residual blood volume fraction. For interpreation of Brown et al 1997,
data.rat.fOWexp.res(adi) = 0.005; % Table 30, see 2nd paragraph on p.457 under BLOOD VOLUME DATA
data.rat.fOWexp.res(bon) = 0.019; % Brown etal 1997(Table 30) = 0.04
data.rat.fOWexp.res(bra) = 0.014; % Brown etal 1997(Table 30) = [0.02-0.04]
data.rat.fOWexp.res(gut) = 0.010;
data.rat.fOWexp.res(hea) = 0.061; % Brown etal 1997(Table 30) = 0.26
data.rat.fOWexp.res(kid) = 0.046; % Brown etal 1997(Table 30) = [0.11-0.27], mean 0.16
data.rat.fOWexp.res(liv) = 0.057; % Brown etal 1997(Table 30) = [0.12-0.27], mean 0.21
data.rat.fOWexp.res(lun) = 0.175; % Brown etal 1997(Table 30) = [0.26-0.52], mean 0.36
data.rat.fOWexp.res(mus) = 0.004; % Brown etal 1997(Table 30) = [0.01-0.04], mean 0.04
data.rat.fOWexp.res(ski) = 0.002; % Brown etal 1997(Table 30) = 0.02
data.rat.fOWexp.res(spl) = 0.321; % Brown etal 1997(Table 30) = [0.17-0.28], mean 0.22


%%% ========================================================================================================
%%% Data  : Fraction of total blood volume that is regional peripheral blood volume
%%%         associated with each organ
%%%
%%% Unit  : percent (%)
%%% Source: see above
%%%

for a = 1:length(rat.subtype)
    
    subtype = rat.subtype{a};
    
    %%% assign fractions of organ weights and tissue volumes
    rat.(subtype).fVtis  = data.rat.fVtis;
    rat.(subtype).fVtot  = data.rat.fVtot;
    
    %%% determine peripheral vascular blood volume associated with each organ
    rat.(subtype).V.vas  = data.rat.fOWexp.vas.*rat.(subtype).OW.exp;
    
    %%% determine total tissue volume
    rat.(subtype).V.tot  = rat.(subtype).V.vas + rat.(subtype).V.tis;
    
    %%% define fraction of total blood that is regional blood, i.e., the
    %%% vascular blood associated with each organ
    rat.(subtype).fVblood.reg = rat.(subtype).V.vas / rat.(subtype).V.blood;
    
    %%% determine residual peripheral blood volume associated with each organ
    rat.(subtype).V.res  = data.rat.fOWexp.res.*rat.(subtype).OW.exp;
    
    %%% define fraction of total blood that is residual regional blood
    rat.(subtype).fVblood.res = rat.(subtype).V.res / rat.(subtype).V.blood;
    
    %%% ASSUMPTION:
    %%%
    %%% Assume that veneous and arterial blood are 2/3 and 1/3 of
    %%% total blood and set the hematocrit value to 0.45
    %%%
    rat.(subtype).fVblood.venous   = 2/3;
    rat.(subtype).fVblood.arterial = 1/3;
    %rat.(subtype).hct = 0.45;
    
    fprintf('change of rat hct\n');
    rat.(subtype).hct = 0.43; % Windberger et al
    
    %     fprintf('change of rat hct\n');
    %     rat.(subtype).hct = 0.46; % Hansen et al  assuming hct of 0.46
    %
end;


%%% ========================================================================================================
%%% Data  : cardiac output and regional blood flows
%%%
%%% Unit  : co in [L/min] and fraction of co that is regional blood flow
%%% Source: Brown et al, Tox Ind Health 1997 (Table 25)
%%%
for a = 1:length(rat.subtype)
    
    subtype = rat.subtype{a};
    rat.(subtype).co = 0.235*(rat.(subtype).BW^0.75); % Brown et al, Tox Ind Health 1997, p.441
    
end;

%%% fraction of cardiac output that is regional blood flow
%%%
data.rat.fQco = T.initialize.tissueDB.NaN;
data.rat.fQco(adi) =  7.0/100;
data.rat.fQco(bon) = 12.2/100;
data.rat.fQco(bra) =  2.0/100;
data.rat.fQco(gut) = 13.1/100; % Poulin & Theil, J Pharm Sci 2002
data.rat.fQco(hea) =  4.9/100;
data.rat.fQco(kid) = 14.1/100;
data.rat.fQco(liv) = 17.4/100;
data.rat.fQco(mus) = 27.8/100;
data.rat.fQco(ski) =  5.8/100;
data.rat.fQco(spl) =  2.0/100; % Poulin & Theil, J Pharm Sci 2002


for a = 1:length(rat.subtype)
    
    subtype = rat.subtype{a};
    rat.(subtype).fQco = data.rat.fQco;
    rat.(subtype).Q.blo = rat.(subtype).fQco * rat.(subtype).co;
end;


%%% ========================================================================================================
%%% Data  : Fraction of tissue volume that is total tissue water
%%%
%%% Unit  : fraction of tissue volume
%%% Source: Rodgers and Rowland, J Pharm Sci (2006), Table 1, total tissue water is reported, corrected
%%%         for residual (see eq. (A2)). According to email correspondance
%%%         with T. Rodgers, f_residual was taken from ref. 15 (Kawai et
%%%         al. 1994), not from ref. 16 as mentioned in the article)
%%%
data.rat.fVtis.wtot = T.initialize.tissueDB.NaN;  %%% total tissue water (extra-cellular)
data.rat.fVtis.wtot(adi) = 0.144;
data.rat.fVtis.wtot(bon) = 0.417;
data.rat.fVtis.wtot(bra) = 0.753;
data.rat.fVtis.wtot(gut) = 0.738;
data.rat.fVtis.wtot(hea) = 0.568;
data.rat.fVtis.wtot(kid) = 0.672;
data.rat.fVtis.wtot(liv) = 0.642;
data.rat.fVtis.wtot(lun) = 0.574;
data.rat.fVtis.wtot(mus) = 0.726;
data.rat.fVtis.wtot(ski) = 0.658;
data.rat.fVtis.wtot(spl) = 0.562;


%%% ========================================================================================================
%%% Data  : Fraction of experimental orgen weight that is extracellular and intracelluar tissue water
%%%
%%% Unit  : fraction of experimental organ weight and tissue volume
%%% Source: Rodgers, Leahy, and Rowland, J Pharm Sci (2005), identical to Graham etal, J Pharm Pharmacol (2011)
%%% Note  : fraction of intra-cellular water was determined as difference
%%%         of extra-cellular water to total experimental tissue water, i.e
%%%         fVexp.wic = fVexp.wtot - fVexp.wex
%%%
data.rat.fVexp.wex = T.initialize.tissueDB.NaN;  %%% tissue water (extra-cellular)
data.rat.fVexp.wex(adi) = 0.135;
data.rat.fVexp.wex(bon) = 0.100;
data.rat.fVexp.wex(bra) = 0.162;
data.rat.fVexp.wex(gut) = 0.282;
data.rat.fVexp.wex(hea) = 0.320;
data.rat.fVexp.wex(kid) = 0.273;
data.rat.fVexp.wex(liv) = 0.161;
data.rat.fVexp.wex(lun) = 0.336;
data.rat.fVexp.wex(mus) = 0.118;
data.rat.fVexp.wex(ski) = 0.382;
data.rat.fVexp.wex(spl) = 0.207;
data.rat.fVexp.wex(pla) = NaN;
data.rat.fVexp.wex(ery) = 0;
%%%
data.rat.fVexp.wic = T.initialize.tissueDB.NaN;  %%% tissue water (intra-cellular)
data.rat.fVexp.wic(adi) = 0.017;
data.rat.fVexp.wic(bon) = 0.346;
data.rat.fVexp.wic(bra) = 0.620;
data.rat.fVexp.wic(gut) = 0.475;
data.rat.fVexp.wic(hea) = 0.456;
data.rat.fVexp.wic(kid) = 0.483;
data.rat.fVexp.wic(liv) = 0.573;
data.rat.fVexp.wic(lun) = 0.446;
data.rat.fVexp.wic(mus) = 0.630;
data.rat.fVexp.wic(ski) = 0.291;
data.rat.fVexp.wic(spl) = 0.579;
data.rat.fVexp.wic(pla) = NaN;
data.rat.fVexp.wic(ery) = 0.603;

%%% ASSUMPTION:
%%%
%%% As stated above, fVexp.wic = fVexp.wtot - fVexp.wex.
%%% Assuming that both, fVexp.wtot and fVexp.wex are perturbed by
%%% water in residual blood and to the same extent, residual blood
%%% perturbations cancel out in fVexp.wic. Since Vexp is approximately equal
%%% to Vtis, we assume that fVtis.wic = fVexp.wic. We then determine again
%%% fVtis.wex by the difference of fVtis.wic to fVtis.wtot
%%%
data.rat.fVtis.wic = data.rat.fVexp.wic;
data.rat.fVtis.wex = data.rat.fVtis.wtot - data.rat.fVtis.wic;


%%% ========================================================================================================
%%% Data  : Fraction of tissue volume that is neutral lipids (nlt) and neutal phospholipids (npt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: Rodgers and Rowland, J Pharm Sci (2006)
%%% Note  : Values have been corrected for residual blood contributions,
%%%         i.e., reported values do not contain resdiual blood contributions
%%%         (see Rodgers & Rowland, p.1252 top left paragraph)
%%%
data.rat.fVtis.nlt = T.initialize.tissueDB.NaN;  %%% neutral lipids
data.rat.fVtis.nlt(adi) = 0.8530; % note: in Rodgers and Rowland (2006) incorrectly reported under neutral phospholipids
data.rat.fVtis.nlt(bon) = 0.0174;
data.rat.fVtis.nlt(bra) = 0.0391;
data.rat.fVtis.nlt(gut) = 0.0375;
data.rat.fVtis.nlt(hea) = 0.0135;
data.rat.fVtis.nlt(kid) = 0.0121;
data.rat.fVtis.nlt(liv) = 0.0135;
data.rat.fVtis.nlt(lun) = 0.0215;
data.rat.fVtis.nlt(mus) = 0.0100;
data.rat.fVtis.nlt(ski) = 0.0603;
data.rat.fVtis.nlt(spl) = 0.0071;
data.rat.fVtis.nlt(pla) = 0.0023; % p. 1241, paragarph "Tissue Specific Input Parameters"
data.rat.fVtis.nlt(ery) = 0.0017; % Rodgers, Leahy, Rowland (2005), Table 1
%%%
data.rat.fVtis.npt = T.initialize.tissueDB.NaN;  %%% neutral phospholipids
data.rat.fVtis.npt(adi) = 0.0016; % note: in Rodgers and Rowland (2006) incorrectly reported under neutral lipids
data.rat.fVtis.npt(bon) = 0.0016;
data.rat.fVtis.npt(bra) = 0.0015;
data.rat.fVtis.npt(gut) = 0.0124;
data.rat.fVtis.npt(hea) = 0.0106;
data.rat.fVtis.npt(kid) = 0.0240;
data.rat.fVtis.npt(liv) = 0.0238;
data.rat.fVtis.npt(lun) = 0.0123;
data.rat.fVtis.npt(mus) = 0.0072;
data.rat.fVtis.npt(ski) = 0.0044;
data.rat.fVtis.npt(spl) = 0.0107;
data.rat.fVtis.npt(pla) = 0.0013; % p. 1241, paragarph "Tissue Specific Input Parameters"
data.rat.fVtis.npt(ery) = 0.0029; % Rodgers, Leahy, Rowland (2005), Table 1


%%% ========================================================================================================
%%% Data  : Intra-cellular acidic phospholipids (APmt) in rat
%%%
%%% Unit  : mg/g tissue (original) scaled to fraction
%%% Source: Rodgers, Leahy, and Rowland, J Pharm Sci (2005), identical to Graham etal, J Pharm Pharmacol (2011)
%%% Note  : Not so clear, whether fractions are with respect to residual
%%%         blood corrected or contaminated tissue weight. We assume that
%%%         values have been corrected for residual blood (as fVtis.nlt or
%%%         fVtis.npt in the same article)
%%%
data.rat.fOWtis.APmt = T.initialize.tissueDB.NaN;
data.rat.fOWtis.APmt(adi) = 0.40;
data.rat.fOWtis.APmt(bon) = 0.67;
data.rat.fOWtis.APmt(bra) = 0.40;
data.rat.fOWtis.APmt(gut) = 2.41;
data.rat.fOWtis.APmt(hea) = 2.25;
data.rat.fOWtis.APmt(kid) = 5.03;
data.rat.fOWtis.APmt(liv) = 4.56;
data.rat.fOWtis.APmt(lun) = 3.91;
data.rat.fOWtis.APmt(mus) = 1.53;
data.rat.fOWtis.APmt(ski) = 1.32;
data.rat.fOWtis.APmt(spl) = 3.18;
data.rat.fOWtis.APmt(pla) = NaN;
data.rat.fOWtis.APmt(ery) = 0.5;

%%% ASSUMPTION:
%%%
%%% scaling from weight fraction mg/g to weight fraction 10e-3 g/g tissue
%%% that are identified with the volume fraction
data.rat.fVtis.APmt = data.rat.fOWtis.APmt/1000;


%%% ========================================================================================================
%%% Data  : Albumin (Atp) in rat
%%%
%%% Unit  : Tissue-to-plasma ratio
%%% Source: Rodgers, and Rowland, J Pharm Sci 95 (2006), identical to Graham etal, J Pharm Pharmacol (2011)
%%% Note  : Not so clear, whether fractions are with respect to residual
%%%         blood corrected or contaminated tissue weight. We assume that
%%%         values have been corrected for residual blood (as fVtis.nlt or
%%%         fVtis.npt in the same article)
%%%
data.rat.r.Atp  = T.initialize.tissueDB.NaN;
data.rat.r.Atp(adi) = 0.049;
data.rat.r.Atp(bon) = 0.100;
data.rat.r.Atp(bra) = 0.048;
data.rat.r.Atp(gut) = 0.158;
data.rat.r.Atp(hea) = 0.157;
data.rat.r.Atp(kid) = 0.130;
data.rat.r.Atp(liv) = 0.086;
data.rat.r.Atp(lun) = 0.212;
data.rat.r.Atp(mus) = 0.064;
data.rat.r.Atp(ski) = 0.277;
data.rat.r.Atp(spl) = 0.097;


%%% ========================================================================================================
%%% Data  : Lipoprotein (Ltp) in rat
%%%
%%% Unit  : Tissue-to-plasma ratio
%%% Source: Rodgers, and Rowland, J Pharm Sci 95 (2006), identical to Graham etal, J Pharm Pharmacol (2011)
%%%
data.rat.r.Ltp = T.initialize.tissueDB.NaN;
data.rat.r.Ltp(adi) = 0.068;
data.rat.r.Ltp(bon) = 0.050;
data.rat.r.Ltp(bra) = 0.041;
data.rat.r.Ltp(gut) = 0.141;
data.rat.r.Ltp(hea) = 0.160;
data.rat.r.Ltp(kid) = 0.137;
data.rat.r.Ltp(liv) = 0.161;
data.rat.r.Ltp(lun) = 0.168;
data.rat.r.Ltp(mus) = 0.059;
data.rat.r.Ltp(ski) = 0.096;
data.rat.r.Ltp(spl) = 0.207;

%%% ========================================================================================================
%%% Model : Assignment of fractional volumes and ratio to all age classes
%%%
for a = 1:length(rat.subtype)
    
    subtype = rat.subtype{a};
    rat.(subtype).fVtis = data.rat.fVtis;
    rat.(subtype).r     = data.rat.r;
    
end;


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++ RAT +++++++
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% ========================================================================================================
%%% Data: Mostly reported for B6C3F1 mice
%%%
%%% Source: Brown et al, Tox Ind Health 1997 (Table 1)
%%%
%%% Note: Male mice of 25g are only 9 weeks of age and in rapid growth phase.
%%% Growth is much slower between age 20-67 weeks (ca. 31.5-40 g) and reaches
%%% a plateau of 40g at age 68-91 weeks.

% mouse25 = 1; mouse40 = 2; mouseBax = 3; mouseFaH = 4;

mouse.type    = 'mouse';
mouse.subtype = {'mouse25','mouse40','mouseBax','mouseFaH'};


%%% ========================================================================================================
%%% Data  : Body weight (BW)
%%%
%%% Units : BW in kg,
%%% Source: Brown et al, Tox Ind Health 1997 (Table 1)
%%% Strain: most data were reported for the B6C3F1 strain

data.mouse.sex =   {'NaN','NaN', 'NaN', 'NaN'};
data.mouse.age =   [    9    70   NaN    NaN ]; % in weeks

data.mouse.BW  =   [0.025 0.040 0.022  0.025 ]; % in kg

%%% Assign values to all age/sex classes
for a = 1:length(mouse.subtype)
    
    subtype = mouse.subtype{a};
    mouse.(subtype)         = struct(data.subtype);
    mouse.(subtype).type    = 'mouse';
    mouse.(subtype).info    = info;
    mouse.(subtype).T       = T;
    mouse.(subtype).age     = data.mouse.age(a);
    mouse.(subtype).sex     = data.mouse.sex{a};
    mouse.(subtype).BW      = data.mouse.BW(a);
    mouse.(subtype).BH      = NaN;
    mouse.(subtype).LBW     = NaN;
    mouse.(subtype).BMI     = NaN;
    mouse.(subtype).BSA     = NaN;
    
end;


%%% ========================================================================================================
%%% Data  : Density of tissue
%%%
%%% Unit  : kg/l (= g/cm^3)
%%%

%%% ASSUMPTION:
%%%
%%% Density assumed identical to human density data
%%% Source: Human data. See Brown et al, Tox Ind Health 1997 and ICRP report 1975
%%%
data.mouse.dens      = ones(size(T.initialize.tissueDB.NaN)); % all organs except for adi and bon (see Brown et al, Tox Ind Health 1997, Table 19)
data.mouse.dens(adi) = 0.916;                  % ICRP report 1975, p.44
data.mouse.dens(bon) = 1.3;                    % ICRP report 2002, Table 2.20 (whole skeleton, adults)


%%% ========================================================================================================
%%% Data  : fraction of total body weight that is experimental organ weight or
%%%         total blood volume
%%%
%%% Unit  : fraction (converted from percentage by dividing by 100]
%%% Source: 1st and 2nd column:
%%%         Brown et al, Tox Ind Health 1997: Table 4 (most tissues),
%%%         Table 10 (adi), Table 21 (adi,blo)
%%%         3rd column:
%%%         Baxter et al, Cancer Research 54 (1994), Table 1
%%%         (volumes were converted to OW with density = 1.0, except
%%%         for bone = 1.5. Subsequently, OW were converted to fOW
%%%         based on BW = 22g, as mentioned in the article (Material and Methods, p.1517).
%%%         fOW_adi and fOW_bra were taken from Brown)
%%%         4th column:
%%%         Mouse from Fronton etal. JPKPD 2014 based on Baxter et al.
%%%
data.mouse.fBW.OWexp.adi = [ 7   /100,  7   /100, 7/100   , 1.749/25]; % 4th column: volume 1.909 converted to weight 1.7486 using a density of 0.916 g/mL, Brown: highly strain and age dependent according to Brown, p.419 (top)
data.mouse.fBW.OWexp.bon = [10.73/100, 10.73/100, 2.250/22, 2.535/25]; % 3rd column: volume (1.5) converted to weight based on dens = 1.5 as used by Baxter, 4th column: volume 1.950 converted to weight 2.5350 using a density of 1.3 g/mL,
data.mouse.fBW.OWexp.bra = [ 1.65/100,  1.65/100, 1.65/100, 1.65/100];
data.mouse.fBW.OWexp.gut = [ 4.22/100,  4.22/100, 3.450/22, 3.871/25]; % 1st & 2nd column: sum of stomach, small and large intestine (4.22=0.60+2.53+1.09)
data.mouse.fBW.OWexp.hea = [ 0.50/100,  0.50/100, 0.133/22, 0.144/25];
data.mouse.fBW.OWexp.kid = [ 1.67/100,  1.67/100, 0.298/22, 0.318/25];
data.mouse.fBW.OWexp.liv = [ 5.49/100,  5.49/100, 0.951/22, 1.024/25];
data.mouse.fBW.OWexp.lun = [ 0.73/100,  0.73/100, 0.191/22, 0.189/25];
data.mouse.fBW.OWexp.mus = [38.40/100, 38.40/100, 7.924/22, 8.948/25];
data.mouse.fBW.OWexp.ski = [16.53/100, 16.53/100, 2.940/22, 3.315/25];
data.mouse.fBW.OWexp.spl = [ 0.35/100,  0.35/100, 0.100/22, 0.089/25];

%data.mouse.fBW.blood     = [ 4.9 /100,  4.9 /100, 1.407/22, 3.046/25]; % 3rd+4th column: plasma volume 0.774(3rd) and 1.675 (4th) converted to blood via V_blo = V_pla/(1-hct)

fprintf('change of mouse Vblood\n');


data.mouse.fBW.blood     = [ 7.2 /100,  7.2 /100, 1.407/22, 3.046/25];      %Insert change 04th Dec 2014


%%72 ml/kg Blood volume Mouse Diehl et al; range of means: 63-80

%%% assign fractional organ weights and tissue volumes
for a = 1:length(mouse.subtype)
    
    subtype = mouse.subtype{a};
    
    mouse.(subtype).fBW.OWexp      = T.initialize.tissueDB.NaN;
    mouse.(subtype).fBW.OWexp(adi) = data.mouse.fBW.OWexp.adi(a);
    mouse.(subtype).fBW.OWexp(bon) = data.mouse.fBW.OWexp.bon(a);
    mouse.(subtype).fBW.OWexp(bra) = data.mouse.fBW.OWexp.bra(a);
    mouse.(subtype).fBW.OWexp(gut) = data.mouse.fBW.OWexp.gut(a);
    mouse.(subtype).fBW.OWexp(hea) = data.mouse.fBW.OWexp.hea(a);
    mouse.(subtype).fBW.OWexp(kid) = data.mouse.fBW.OWexp.kid(a);
    mouse.(subtype).fBW.OWexp(liv) = data.mouse.fBW.OWexp.liv(a);
    mouse.(subtype).fBW.OWexp(lun) = data.mouse.fBW.OWexp.lun(a);
    mouse.(subtype).fBW.OWexp(mus) = data.mouse.fBW.OWexp.mus(a);
    mouse.(subtype).fBW.OWexp(ski) = data.mouse.fBW.OWexp.ski(a);
    mouse.(subtype).fBW.OWexp(spl) = data.mouse.fBW.OWexp.spl(a);
    
    %%% total blood volume, including peripheral regional blood
    mouse.(subtype).fBW.blood      = data.mouse.fBW.blood(a);
    
    %%% experimental organ weights
    mouse.(subtype).OW.exp   = mouse.(subtype).fBW.OWexp * mouse.(subtype).BW;
    mouse.(subtype).OW.blood = mouse.(subtype).fBW.blood * mouse.(subtype).BW;
    
    %%% ASSUMPTION:
    %%%
    %%% We assume that the experimental organ weights (including residual
    %%% blood to some varying degree) are approximately equal to
    %%% the tissue organ weight (not including residual blood), since
    %%% according to Brown et al. p.411, in most cases, the values provided
    %%% reflect the weight of organs that are drained of blood.
    %%%
    mouse.(subtype).OW.tis = mouse.(subtype).OW.exp;
    mouse.(subtype).LBW    = mouse.(subtype).BW  - mouse.(subtype).OW.tis(adi);
    
    mouse.(subtype).dens   = data.mouse.dens;
    mouse.(subtype).V.tis  = mouse.(subtype).OW.tis ./ mouse.(subtype).dens;
    
    %%% total blood volume (assuming a density of 1)
    %%%
    mouse.(subtype).V.blood = mouse.(subtype).OW.blood;
    
end;


%%% ========================================================================================================
%%% Data  : Fraction of experimental organ weight that is vascular volume and
%%%         interstitial space volume
%%%
%%% Unit  : fraction in [L/kg]

%%% ASSUMPTION:
%%%
%%% Data for mice are identical to the corresponding data for rats. This is
%%% supported by Brown et al, Table 30, comparing the data for mouse and
%%% rat, when taking the reported range of values (n=3) into account. Note
%%% that the mean has been reported, although it is very sensitive to
%%% outliers.
data.mouse.fOWexp = data.rat.fOWexp;

%%% ASSUMPTION:
%%%
%%% Data for mice are identical to the corresponding data for rats.
data.mouse.fOWtot = data.rat.fOWtot;


%%% ASSUMPTION:
%%%
%%% Data for mice are identical to the corresponding data for rats.
data.mouse.fVtot  = data.rat.fVtot;

%%% ASSUMPTION:
%%%
%%% Data for mice are identical to the corresponding data for rats.
data.mouse.fVtis  = data.rat.fVtis;


%%% ========================================================================================================
%%% Data  : Fraction of experimental organ weight that is residual blood
%%%
%%% Unit  : volume fraction of experimental organ weight
%%% Source: residual fractions, Garg, PhD thesis (2007), Table III, p.105,
%%%         reported as ratio of the residual blood volume to the tissue volume in mL/100g tissue
%%%

data.mouse.fOWexp.res = T.initialize.tissueDB.NaN; % residual blood fraction. For interpreation of Brown et al 1997,
data.mouse.fOWexp.res(adi) =   NaN/100; % Table 30, see 2nd paragraph on p.457 under BLOOD VOLUME DATA
data.mouse.fOWexp.res(bon) =   NaN/100; % Brown etal 1997(Table 30) = 0.11
data.mouse.fOWexp.res(bra) =   NaN/100; % Brown etal 1997(Table 30) = 0.03
data.mouse.fOWexp.res(gut) =  1.27/100;
data.mouse.fOWexp.res(hea) =  4.81/100;
data.mouse.fOWexp.res(kid) =  6.23/100; % Brown etal 1997(Table 30) = [0.12-0.34], mean 0.24
data.mouse.fOWexp.res(liv) =  5.27/100; % Brown etal 1997(Table 30) = [0.23-0.36], mean 0.31
data.mouse.fOWexp.res(lun) = 13.13/100; % Brown etal 1997(Table 30) = [0.40-0.62], mean 0.50
data.mouse.fOWexp.res(mus) =  0.63/100; % Brown etal 1997(Table 30) = [0.03-0.05], mean 0.04
data.mouse.fOWexp.res(ski) =  0.77/100; % Brown etal 1997(Table 30) = 0.03
data.mouse.fOWexp.res(spl) = 21.51/100; % Brown etal 1997(Table 30) = [0.17-0.19], mean 0.17


%%% ========================================================================================================
%%% Data  : Fraction of total blood volume that is regional peripheral blood volume
%%%         associated with each organ
%%%
%%% Unit  : percent (%)
%%% Source: see above
%%%

for a = 1:length(mouse.subtype)
    
    subtype = mouse.subtype{a};
    
    %%% assign fractions of organ weights and tissue volumes
    mouse.(subtype).fVtis  = data.mouse.fVtis;
    mouse.(subtype).fVtot  = data.mouse.fVtot; %uebergeben ins modell
    
    %%% determine peripheral vascular blood volume associated with each organ
    mouse.(subtype).V.vas  = data.mouse.fOWexp.vas.*mouse.(subtype).OW.exp;
    
    %%% determine total tissue volume
    mouse.(subtype).V.tot  = mouse.(subtype).V.vas + mouse.(subtype).V.tis;
    
    %%% define fraction of total blood that is regional blood, i.e., the
    %%% vascular blood associated with each organ
    mouse.(subtype).fVblood.reg = mouse.(subtype).V.vas / mouse.(subtype).V.blood;
    
    %%% determine residual peripheral blood volume associated with each organ
    mouse.(subtype).V.res  = data.mouse.fOWexp.res.*mouse.(subtype).OW.exp;
    
    %%% define fraction of total blood that is residual regional blood
    mouse.(subtype).fVblood.res = mouse.(subtype).V.res / mouse.(subtype).V.blood;
    
    %%% ASSUMPTION:
    %%%
    %%% Assume that veneous and arterial blood are 2/3 and 1/3 of
    %%% total blood and set the hematocrit value to 0.40
    %%%
    mouse.(subtype).fVblood.venous   = 2/3;
    mouse.(subtype).fVblood.arterial = 1/3;
    %mouse.(subtype).hct = 0.45;  %Davies Morries et al
    
    fprintf('change of mouse hct\n');
    mouse.(subtype).hct = 0.40; %Windberger et al                               %Insert change 04th Dec 2014
    
end;


%%% ========================================================================================================
%%% Data  : cardiac output and regional blood flows
%%%
%%% Unit  : co in [L/min] and fraction of co that is regional blood flow
%%% Source: Brown et al, Tox Ind Health 1997 (Table 24) or for the 3rd column
%%%         Baxter et al. (1994) as fraction of plasma flow = fraction of
%%%         blood flow; and El-Masri and Portier, DMD Vol.26 (1998), 585-594
%%%         for adi, bon, bra, gut and spl
%%%
for a = 1:length(mouse.subtype)
    
    subtype = mouse.subtype{a};
    mouse.(subtype).co = 0.275*(mouse.(subtype).BW^0.75);
    
    %%% Brown et al, Tox Ind Health 1997, p.440 (based on Arms and Travis
    %%% (1988)), involving different scaling steps
    
end;
mouse.mouseBax.co = 4.38/(1-0.45)/1000; % converting from plasma co in [mL/min] to blood co in [L/min]
mouse.mouseFaH.co = 4.38/(1-0.45)/1000; % converting from plasma co in [mL/min] to blood co in [L/min]

%%% fraction of cardiac output that is regional blood flow
%%%
%%%                       Brown     Brown    Baxter   Fronton/Hui
data.mouse.fQco.adi = [   7/100,    7/100, 7   /100 , 0.307/4.380];  %  7%   of blood flow, El-Masri (1998)
data.mouse.fQco.bon = [  11/100,   11/100, 0.17/4.38, 0.170/4.380];  % 11%   of blood flow, El-Masri (1998)
data.mouse.fQco.bra = [ 3.3/100,  3.3/100, 3.3 /100 , 3.3  /100  ];  %  3.3% of blood flow, El-Masri (1998)
data.mouse.fQco.gut = [14.1/100, 14.1/100, 0.90/4.38, 0.900/4.380];  % 14.1% of blood flow, El-Masri (1998)
data.mouse.fQco.hea = [ 6.6/100,  6.6/100, 0.28/4.38, 0.280/4.380];
data.mouse.fQco.kid = [ 9.1/100,  9.1/100, 0.80/4.38, 0.800/4.380];
data.mouse.fQco.liv = [16.2/100, 16.2/100, 1.10/4.38, 1.100/4.380];
data.mouse.fQco.mus = [15.9/100, 15.9/100, 0.80/4.38, 0.800/4.380];
data.mouse.fQco.ski = [ 5.8/100,  5.8/100, 1.21/4.38, 1.210/4.380];
data.mouse.fQco.spl = [   1/100,    1/100, 0.05/4.38, 0.050/4.380];  %  1%   of blood flow, El-Masri (1998)

for a = 1:length(mouse.subtype)
    
    subtype = mouse.subtype{a};
    mouse.(subtype).fQco = T.initialize.tissueDB.NaN;
    
    mouse.(subtype).fQco(adi) = data.mouse.fQco.adi(a);
    mouse.(subtype).fQco(bon) = data.mouse.fQco.bon(a);
    mouse.(subtype).fQco(bra) = data.mouse.fQco.bra(a);
    mouse.(subtype).fQco(gut) = data.mouse.fQco.gut(a);
    mouse.(subtype).fQco(hea) = data.mouse.fQco.hea(a);
    mouse.(subtype).fQco(kid) = data.mouse.fQco.kid(a);
    mouse.(subtype).fQco(liv) = data.mouse.fQco.liv(a);
    mouse.(subtype).fQco(mus) = data.mouse.fQco.mus(a);
    mouse.(subtype).fQco(ski) = data.mouse.fQco.ski(a);
    mouse.(subtype).fQco(spl) = data.mouse.fQco.spl(a);
    
    mouse.(subtype).Q.blo = mouse.(subtype).fQco * mouse.(subtype).co;
    
end;


%%% ========================================================================================================
%%% Data  : Total tissue water (Vwt)
%%%
%%% Unit  : fraction of tissue volume
%%%

%%% ASSUMPTION:
%%%
%%% Mouse values are identical to rat values
%%%
data.mouse.fVtis.wex = data.rat.fVtis.wex;
data.mouse.fVtis.wic = data.rat.fVtis.wic;


%%% ========================================================================================================
%%% Data  : Neutral lipids (fVtis.nlt) and neutal phospholipids (fVtis.npt)
%%%
%%% Unit  : fraction of tissue volume
%%%

%%% ASSUMPTION:
%%%
%%% Mouse values are identical to rat values
%%%
data.mouse.fVtis.nlt = data.rat.fVtis.nlt;
data.mouse.fVtis.npt = data.rat.fVtis.npt;


%%% ========================================================================================================
%%% Data  : Intra-cellular acidic phospholipids (APmt) in rat
%%%
%%% Unit  : mg/g tissue (original) scalled to fraction
%%%

%%% ASSUMPTION:
%%%
%%% Mouse values are identical to rat values
%%%
data.mouse.fVtis.APmt = data.rat.fVtis.APmt;


%%% ========================================================================================================
%%% Data  : Albumin (Atp) in rat
%%%
%%% Unit  : Tissue-to-plasma ratio
%%%

%%% ASSUMPTION:
%%%
%%% Mouse values are identical to rat values
%%%
data.mouse.r.Atp  = data.rat.r.Atp;


%%% ========================================================================================================
%%% Data  : Lipoprotein (Ltp) in rat
%%%
%%% Unit  : Tissue-to-plasma ratio
%%%

%%% ASSUMPTION:
%%%
%%% Mouse values are identical to rat values
%%%
data.mouse.r.Ltp = data.rat.r.Ltp;

%%% ========================================================================================================
%%% Model : Assignment of fractional volumes and ratio to all age classes
%%%
for a = 1:length(mouse.subtype)
    
    subtype = mouse.subtype{a};
    mouse.(subtype).fVtis = data.mouse.fVtis;
    mouse.(subtype).r     = data.mouse.r;
    
end;


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++ MOUSE +++++++
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% ========================================================================================================
%%% Model: Age and sex
%%%
%%% Children:
%%% newborn (nb), age 1 (age1), 5 (age5), 10 (age10) , all uni sex
%%% age 15 male (age15m), age 15 female (age15f),
%%% Adults:
%%% age 20-50 male (age35m), age 20-50 female (age35f)
%%% As in the source, we associate an average age of 35 with the adult
%%%
%%% Source: ICRP report 2002
%%%
human.type    = 'human';
human.subtype = {'human0u','human1u','human5u','human10u','human15m','human15f','human35m','human35f'};


%%% ========================================================================================================
%%% Data  : Body weight (BW), body height (BH) and body surface area (BSA)
%%%
%%% Units : BW in kg, BH in cm, BSA in m^2
%%% Source: ICRP report 2002 (Table 2.9)
%%%
%%%                 nb   age1    age5   age10  age15m  age15f  age35m   age35f
ICRP.BW  =      [ 3.50  10.00   19.00   32.00   56.00   53.00   73.00   60.00];
ICRP.BH  =      [51.00  76.00  109.00  138.00  167.00  161.00  176.00  163.00];
ICRP.BSA =      [ 0.24   0.48    0.78    1.12    1.62    1.55    1.90    1.66];
ICRP.sex =      {'uni'  'uni'   'uni'   'uni'  'male' 'female' 'male' 'female'};
ICRP.age =      [    0      1       5      10     15       15      35      35];

%%% Assign values to all age/sex classes
for a = 1:length(human.subtype)
    
    age = human.subtype{a};
    human.(age)         = struct(data.subtype);
    human.(age).type    = 'human';
    human.(age).info    = info;
    human.(age).T       = T;
    human.(age).age     = ICRP.age(a);
    human.(age).sex     = ICRP.sex{a};
    human.(age).BW      = ICRP.BW(a);
    human.(age).BH      = ICRP.BH(a)/100;
    human.(age).BSA     = ICRP.BSA(a);
    human.(age).BMI     = human.(age).BW/human.(age).BH^2;
    
end;


%%% ========================================================================================================
%%% Data  : Organ weight, density and volume
%%%
%%% Unit  : weight in g, density in g/mL, volume in L
%%% Source: ICRP report 2002 (Table 2.8)
%%%
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.OW.exp.adipose_tissue =                       [930   3800   5500   8600   12000   18700   18200   22500]; % entry duplicates other mass information
ICRP.OW.exp.separable_adipose =                    [890   3600   5000   7500    9500   16000   14500   19000];
ICRP.OW.exp.adrenals =                             [  6      4      5      7      10       9      14      13];
% alimentary system
ICRP.OW.exp.tongue =                               [  3.5   10     19     32      56      53      73      60];
ICRP.OW.exp.salivary_glands =                      [  6     24     34     44      68      65      85      70];
ICRP.OW.exp.oesophagus_wall =                      [  2      5     10     18      30      30      40      35];
ICRP.OW.exp.stomach_wall =                         [  7     20     50     85     120     120     150     140];
ICRP.OW.exp.stomach_contents =                     [ 40     67     83    117     200     200     250     230];
ICRP.OW.exp.small_intestine_wall =                 [ 30     85    220    370     520     520     650     600];
ICRP.OW.exp.small_intestine_contents =             [ 56     93    117    163     280     280     350     280];
ICRP.OW.exp.large_intestine_right_colon_wall =     [  7     20     49     85     122     122     150     145];
ICRP.OW.exp.large_intestine_right_colon_contents = [ 24     40     50     70     120     120     150     160];
ICRP.OW.exp.large_intestine_left_colon_wall =      [  7     20     49     85     122     122     150     145];
ICRP.OW.exp.large_intestine_left_colon_contents =  [ 12     20     25     35      60      60      75      80];
ICRP.OW.exp.large_intestine_rectosigmoid_wall =    [  3     10     22     40      56      56      70      70];
ICRP.OW.exp.large_intestine_rectosigmoid_content = [ 12     20     25     35      60      60      75      80];
ICRP.OW.exp.liver =                                [130    330    570    830    1300    1300    1800    1400];
ICRP.OW.exp.gallbladder_wall =                     [  0.5    1.4    2.6    4.4     7.7     7.3    10       8];
ICRP.OW.exp.gallbladder_content =                  [  2.8    8     15     26      45      42      58      48];
ICRP.OW.exp.pancreas =                             [  6     20     35     60     110     100     140     120];

%%% Note: brain organ weight age5  = 1245 = mean value of 1310 (male) and 1180 (female)
%%%       brain organ weight age10 = 1310 = mean value of 1400 (male) and 1220 (female)
ICRP.OW.exp.brain =                                [380    950   1245   1310    1420    1300    1450    1300];
ICRP.OW.exp.breasts =                              [NaN    NaN    NaN    NaN      15     250      25     500];
% circulatory system
ICRP.OW.exp.heart_with_blood =                     [ 46     98    220    370     660     540     840     620]; % entry duplicates other mass information
ICRP.OW.exp.heart_tissue_only =                    [ 20     50     85    140     230     220     330     250];
% Table sec 7.4
ICRP.OW.exp.total_blood =                          [270    500   1400   2400    4500    3300    5300    3900];
ICRP.OW.exp.eyes =                                 [  6      7     11     12      13      13      15      15];
ICRP.OW.exp.storage_fat =                          [370   2300   3600   6000    9000   14000   14600   18000]; % entry duplicates other mass information
ICRP.OW.exp.skin =                                 [175    350    570    820    2000    1700    3300    2300];
ICRP.OW.exp.skeletal_muscle =                      [800   1900   5600  11000   24000   17000   29000   17500];
ICRP.OW.exp.pituitary_gland =                      [  0.1   0.15   0.25   0.35     0.5     0.5     0.6     0.6];
% Respiratory system
ICRP.OW.exp.larynx =                               [  1.3    4      7     12      22      15      28      19];
ICRP.OW.exp.trachea =                              [  0.5    1.5    2.5    4.5     7.5     6      10       8];
ICRP.OW.exp.lung_with_blood =                      [ 60    150    300    500     900     750    1200     950]; % entry duplicates other mass information
ICRP.OW.exp.lung_tissue_only =                     [ 30     80    125    210     330     290     500     420];
% skeletal system
ICRP.OW.exp.total_skeleton =                       [370   1170   2430   4500    7950    7180   10500    7800]; % entry duplicates other mass information
ICRP.OW.exp.bone_cortical =                        [135    470   1010   1840    3240    2960    4400    3200];
ICRP.OW.exp.bone_trabecular =                      [ 35    120    250    460     810     740    1100     800];
% bone_total = bone_cortical + bone_trabecular
ICRP.OW.exp.bone_total =                           [170    590   1260   2300    4050    3700    5500    4000]; % entry duplicates other mass information
ICRP.OW.exp.marrow_active =                        [ 50    150    340    630    1080    1000    1170     900];
ICRP.OW.exp.marrow_inactive =                      [  0     20    160    630    1480    1380    2480    1800];
ICRP.OW.exp.cartilage =                            [130    360    600    820    1140     920    1100     900];
ICRP.OW.exp.teeth =                                [  0.7    5     15     30      45      35      50      40];
ICRP.OW.exp.skeleton_miscellaneous =               [ 20     45     55     90     155     145     200     160];
ICRP.OW.exp.spleen =                               [  9.5   29     50     80     130     130     150     130];


%%% Note: thymus organ weight age10 = 37.5 = mean value of 40 (male) and 35 (female)
ICRP.OW.exp.thymus =                               [ 13     30     30     37.5    35      30      25      20];
ICRP.OW.exp.thyroid =                              [  1.3    1.8    3.4    7.9    12      12      20      17];
ICRP.OW.exp.tonsils =                              [  0.1    0.5    2      3       3       3        3      3];
% Urogenital system
ICRP.OW.exp.kidneys =                              [ 25     70    110    180     250     240     310     275];
ICRP.OW.exp.ureters =                              [  0.77   2.2    4.2    7.0    12      12      16      15];
ICRP.OW.exp.urinary_bladder =                      [  4      9     16     25      40      35      50      40];

%%% Note: urethra organ weight nb    = 0.31 = mean value of 0.48 (male) and 0.14 (female)
%%%       urethra organ weight age1  = 0.91 = mean value of 1.4  (male) and 0.42 (female)
%%%       urethra organ weight age5  = 1.69 = mean value of 2.6  (male) and 0.78 (female)
%%%       urethra organ weight age10 = 2.85 = mean value of 4.4  (male) and 1.3  (female)
ICRP.OW.exp.urethra =                              [  0.31   0.91   1.69   2.85    7.7     2.3    10       3];
ICRP.OW.exp.testes =                               [  0.85   1.5    1.7    2      16       0      35       0];
ICRP.OW.exp.epididymes =                           [  0.25   0.35   0.45   0.6     1.6     0       4       0];
ICRP.OW.exp.prostate =                             [  0.8    1.0    1.2    1.6     4.3     0      17       0];
ICRP.OW.exp.ovaries =                              [  0.3    0.8    2.0    3.5     0       6       0      11];
ICRP.OW.exp.fallopian_tubes =                      [  0.25   0.25   0.35   0.50    0       1.1     0     2.1];
ICRP.OW.exp.uterus =                               [  4.0    1.5    3      4       0      30       0      80];


%%% ========================================================================================================
%%% Data  : Regional blood volumes as percentage of total blood volume
%%%
%%% Unit  : percent (%)
%%% Source: ICRP report 2002 (Table 2.14)
%%%
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.fVblood.fat =                             [NaN    NaN    NaN    NaN     NaN     NaN    5.00    8.50];
ICRP.fVblood.brain =                           [NaN    NaN    NaN    NaN     NaN     NaN    1.20    1.20];
ICRP.fVblood.stomach_and_oesophages =          [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00];
ICRP.fVblood.small_intestine =                 [NaN    NaN    NaN    NaN     NaN     NaN    3.80    3.80];
ICRP.fVblood.large_intestine =                 [NaN    NaN    NaN    NaN     NaN     NaN    2.20    2.20];
ICRP.fVblood.right_heart =                     [NaN    NaN    NaN    NaN     NaN     NaN    4.50    4.50];
ICRP.fVblood.left_heart =                      [NaN    NaN    NaN    NaN     NaN     NaN    4.50    4.50];
ICRP.fVblood.coronary_tissue =                 [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00];
ICRP.fVblood.kidneys =                         [NaN    NaN    NaN    NaN     NaN     NaN    2.00    2.00];
ICRP.fVblood.liver =                           [NaN    NaN    NaN    NaN     NaN     NaN   10.00   10.00];
ICRP.fVblood.bronchial_tissue =                [NaN    NaN    NaN    NaN     NaN     NaN    2.00    2.00];
ICRP.fVblood.skeletal_muscle =                 [NaN    NaN    NaN    NaN     NaN     NaN   14.00   10.50];
ICRP.fVblood.pancreas =                        [NaN    NaN    NaN    NaN     NaN     NaN    0.60    0.60];
ICRP.fVblood.skeleton.total =                  [NaN    NaN    NaN    NaN     NaN     NaN    7.00    7.00];
ICRP.fVblood.skeleton.red_marrow =             [NaN    NaN    NaN    NaN     NaN     NaN    4.00    4.00]; % entry duplicates other information
ICRP.fVblood.skeleton.trabecular_bone =        [NaN    NaN    NaN    NaN     NaN     NaN    1.20    1.20]; % entry duplicates other information
ICRP.fVblood.skeleton.cortical_bone =          [NaN    NaN    NaN    NaN     NaN     NaN    0.80    0.80]; % entry duplicates other information
ICRP.fVblood.skeleton.others =                 [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00]; % entry duplicates other information
ICRP.fVblood.skin =                            [NaN    NaN    NaN    NaN     NaN     NaN    3.00    3.00];
ICRP.fVblood.spleen =                          [NaN    NaN    NaN    NaN     NaN     NaN    1.40    1.40];
ICRP.fVblood.thyroid =                         [NaN    NaN    NaN    NaN     NaN     NaN    0.06    0.06];
ICRP.fVblood.lymph_nodes =                     [NaN    NaN    NaN    NaN     NaN     NaN    0.20    0.20];
ICRP.fVblood.gonads =                          [NaN    NaN    NaN    NaN     NaN     NaN    0.04    0.02];
ICRP.fVblood.adrenals =                        [NaN    NaN    NaN    NaN     NaN     NaN    0.06    0.06];
ICRP.fVblood.urinary_bladder =                 [NaN    NaN    NaN    NaN     NaN     NaN    0.02    0.02];
ICRP.fVblood.all_other_tissues =               [NaN    NaN    NaN    NaN     NaN     NaN    1.92    1.92];


%%% ========================================================================================================
%%% Data  : Volume of blood plasma and red blood cells
%%%
%%% Unit  : ml
%%% Source: ICRP report (Table 2.12)
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.erythrocyte_volume =                      [NaN    NaN    NaN    NaN     NaN     NaN    2300    1500];
ICRP.plasma_volume =                           [NaN    NaN    NaN    NaN     NaN     NaN    3000    2400];


%%% ========================================================================================================
%%% Data  : Distribution of blood in the vascular system
%%%
%%% Unit  : Percentage (%) of total blood volume
%%% Source: ICRP report 2002 (Table 2.13)
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.fVblood.heart_chambers =                  [NaN    NaN    NaN    NaN     NaN     NaN    9.00     NaN];
ICRP.fVblood.pulmonary.total =                 [NaN    NaN    NaN    NaN     NaN     NaN   10.50     NaN];
ICRP.fVblood.pulmonary.arteries =              [NaN    NaN    NaN    NaN     NaN     NaN    3.00     NaN];
ICRP.fVblood.pulmonary.capillaries =           [NaN    NaN    NaN    NaN     NaN     NaN    2.00     NaN];
ICRP.fVblood.pulmonary.veins =                 [NaN    NaN    NaN    NaN     NaN     NaN    5.50     NaN];
ICRP.fVblood.systemic.total =                  [NaN    NaN    NaN    NaN     NaN     NaN   80.50     NaN];
ICRP.fVblood.systemic.aorta_large_arteries =   [NaN    NaN    NaN    NaN     NaN     NaN    6.00     NaN];
ICRP.fVblood.systemic.small_arteries =         [NaN    NaN    NaN    NaN     NaN     NaN   10.00     NaN];
ICRP.fVblood.systemic.capillaries =            [NaN    NaN    NaN    NaN     NaN     NaN    5.00     NaN];
ICRP.fVblood.systemic.small_veins =            [NaN    NaN    NaN    NaN     NaN     NaN   41.50     NaN];
ICRP.fVblood.systemic.large_veins =            [NaN    NaN    NaN    NaN     NaN     NaN   18.00     NaN];


%%% ========================================================================================================
%%% Data  : Density of tissue
%%%
%%% Unit  : kg/l (= g/cm^3)
%%% Source: Brown et al, Tox Ind Health 1997 and ICRP report 1975
%%%
data.human.dens      = ones(size(T.initialize.tissueDB.NaN)); % all organs except for adi and bon (see Brown et al, Tox Ind Health 1997, Table 19)
data.human.dens(adi) = 0.916;                  % ICRP report 1975, p.44
data.human.dens(bon) = 1.3;                    % ICRP report 2002, Table 2.20 (whole skeleton, adults)


%%% ========================================================================================================
%%% Model: Regional vascular blood volumes and fractions of total blood
%%%
%%% Unit : [L] and fraction of total blood
%%%
for a = 1:length(human.subtype)
    
    age = human.subtype{a};
    human.(age).fVblood.reg      = T.initialize.tissueDB.NaN;
    human.(age).fVblood.reg(adi) = ICRP.fVblood.fat(a);
    human.(age).fVblood.reg(bon) = ICRP.fVblood.skeleton.total(a);
    human.(age).fVblood.reg(bra) = ICRP.fVblood.brain(a);
    human.(age).fVblood.reg(gut) = ICRP.fVblood.small_intestine(a)+ICRP.fVblood.large_intestine(a);
    human.(age).fVblood.reg(hea) = ICRP.fVblood.coronary_tissue(a);
    human.(age).fVblood.reg(kid) = ICRP.fVblood.kidneys(a);
    human.(age).fVblood.reg(liv) = ICRP.fVblood.liver(a);
    human.(age).fVblood.reg(lun) = ICRP.fVblood.bronchial_tissue(a);
    human.(age).fVblood.reg(mus) = ICRP.fVblood.skeletal_muscle(a);
    human.(age).fVblood.reg(ski) = ICRP.fVblood.skin(a);
    human.(age).fVblood.reg(spl) = ICRP.fVblood.spleen(a);
    
    
    human.(age).fVblood.arterial = 0.5*ICRP.fVblood.heart_chambers(a) +...
        ICRP.fVblood.pulmonary.veins(a) + ...
        0.5*ICRP.fVblood.pulmonary.capillaries(a) +...
        ICRP.fVblood.systemic.aorta_large_arteries(a) + ...
        0.5*ICRP.fVblood.systemic.capillaries(a) + ...
        ICRP.fVblood.systemic.small_arteries(a);
    human.(age).fVblood.venous   = 0.5*ICRP.fVblood.heart_chambers(a) + ...
        ICRP.fVblood.pulmonary.arteries(a) +...
        0.5*ICRP.fVblood.pulmonary.capillaries(a) + ...
        0.5*ICRP.fVblood.systemic.capillaries(a) +...
        ICRP.fVblood.systemic.small_veins(a) + ...
        ICRP.fVblood.systemic.large_veins(a);
    
    %%% change of units: percentage -> fraction
    human.(age).fVblood.reg      = human.(age).fVblood.reg/100;
    human.(age).fVblood.venous   = human.(age).fVblood.venous/100;
    human.(age).fVblood.arterial = human.(age).fVblood.arterial/100;
    
    human.(age).hct              = ICRP.erythrocyte_volume(a)/ICRP.OW.exp.total_blood(a);
    
end;

%%% ASSUMPTION:
%%%
%%% For female adults, male data for fraction of venous and arterial blood
%%% were adopted
human.human35f.fVblood.venous   = human.human35m.fVblood.venous;
human.human35f.fVblood.arterial = human.human35m.fVblood.arterial;


%%% ASSUMPTION:
%%%
%%% For children of age 1,5, 10 and 15f, adult female data were adopted
%%% for all fVblood entries. For children of age 15m, corresponding
%%% adult male data were adopted. This is in line with the NHANES study
%%% (for age 5 and older)
%%%
%%%
children    = {'human0u', 'human1u', 'human5u' ,'human10u','human15m','human15f'};
refAgeClass = {'human35f','human35f','human35f','human35f','human35m','human35f'};

for a = 1:length(children)
    
    %%% target age (age) and age class used for reference (refAge)
    age        = children{a};
    refAge     = refAgeClass{a};
    
    human.(age).fVblood = human.(refAge).fVblood;
    human.(age).hct     = human.(refAge).hct;
    
end;

%%% ========================================================================================================
%%% Model: Organ weights (OW) and tissue volumes (V)
%%%
%%% Unit : OW in kg and V in L
%%%
for a = 1:length(human.subtype)
    
    age = human.subtype{a};
    %%% orignal data in gram
    human.(age).OW.exp      = T.initialize.tissueDB.NaN;
    human.(age).OW.exp(adi) = ICRP.OW.exp.separable_adipose(a);
    human.(age).OW.exp(bon) = ICRP.OW.exp.total_skeleton(a); % includes total bone and marrow
    human.(age).OW.exp(bra) = ICRP.OW.exp.brain(a);
    human.(age).OW.exp(gut) = ICRP.OW.exp.small_intestine_wall(a) + ICRP.OW.exp.large_intestine_right_colon_wall(a) + ...
        ICRP.OW.exp.large_intestine_left_colon_wall(a) + ICRP.OW.exp.large_intestine_rectosigmoid_wall(a);
    human.(age).OW.exp(hea) = ICRP.OW.exp.heart_tissue_only(a);
    human.(age).OW.exp(kid) = ICRP.OW.exp.kidneys(a);
    human.(age).OW.exp(liv) = ICRP.OW.exp.liver(a);
    human.(age).OW.exp(lun) = ICRP.OW.exp.lung_tissue_only(a);
    human.(age).OW.exp(mus) = ICRP.OW.exp.skeletal_muscle(a);
    human.(age).OW.exp(ski) = ICRP.OW.exp.skin(a);
    human.(age).OW.exp(spl) = ICRP.OW.exp.spleen(a);
    
    human.(age).OW.blood    = ICRP.OW.exp.total_blood(a);
    
    %%% change of units: g -> kg
    human.(age).OW.exp      = human.(age).OW.exp/1000;
    human.(age).OW.blood    = human.(age).OW.blood/1000;
    
    human.(age).fBW.OWexp   = human.(age).OW.exp/human.(age).BW;
    human.(age).fBW.blood   = human.(age).OW.blood/human.(age).BW;
    
    %%% ASSUMPTION:
    %%%
    %%% We assume that the experimental organ weights (including residual
    %%% blood to some varying degree) are approximately equal to
    %%% the tissue organ weight (not including residual blood), since
    human.(age).OW.tis     = human.(age).OW.exp;
    human.(age).LBW         = human.(age).BW - human.(age).OW.tis(adi);
    
    
    %%% density and blood & tissue volumes
    human.(age).dens        = data.human.dens;
    
    human.(age).V.blood     = human.(age).OW.blood/1; % change of units: g -> kg, assuming density = 1
    
    human.(age).V.tis       = human.(age).OW.tis ./ data.human.dens;
    human.(age).V.vas       = human.(age).fVblood.reg * human.(age).V.blood;
    human.(age).V.tot       = human.(age).V.vas + human.(age).V.tis;
    
end;


%%% ========================================================================================================
%%% Data  : Cardiac output
%%%
%%% Unit  : L/min
%%% Source: ICRP report 2002 (Table 2.39). For newborn and children age 1,
%%% teh values of Alverson et al 1987 (cited in Abraham et al, Arch Toxicol
%%% (2005) 79: 63?73) have been taken, since the values of the ICRP report
%%% appear to be too low (age1: ICRP.co=1.2, Alverson.co = 1.8. For newborn
%%% the difference is 0.6 vs. 0.7. The ICRP value for age 1 is a makred
%%% outlier from the expectations of allometric scaling.

%%%           nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.co  =  [0.6    1.2    3.4    5.0     6.1     6.1     6.5     5.9];
ICRP.co([1 2]) = [44   109]/60;

%%% ========================================================================================================
%%% Data  : blood flow rates
%%%
%%% Unit  : percentage (%) of cardiac output
%%% Source: ICRP report 2002 (Table 2.40)
%%%                                              nb   age1   age5  age10  age15m  age15f  age35m   age35f
ICRP.fQco.fat =                                [NaN    NaN    NaN    NaN     NaN     NaN    5.00    8.50];
ICRP.fQco.brain =                              [NaN    NaN    NaN    NaN     NaN     NaN   12.00   12.00];
ICRP.fQco.stomach_and_oesophagus =             [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00]; % not used in PBPK model
ICRP.fQco.small_intestine =                    [NaN    NaN    NaN    NaN     NaN     NaN   10.00   11.00];
ICRP.fQco.large_intestine =                    [NaN    NaN    NaN    NaN     NaN     NaN    4.00    5.00];
ICRP.fQco.coronary_tissue =                    [NaN    NaN    NaN    NaN     NaN     NaN    4.00    5.00];
ICRP.fQco.kidney =                             [NaN    NaN    NaN    NaN     NaN     NaN   19.00   17.00];
ICRP.fQco.liver =                              [NaN    NaN    NaN    NaN     NaN     NaN   25.50   27.00]; % total, arterial = [6.5 6.5]
ICRP.fQco.bronchial_tissue =                   [NaN    NaN    NaN    NaN     NaN     NaN    2.50    2.50]; % not used in PBPK model
ICRP.fQco.skeletal_muscle =                    [NaN    NaN    NaN    NaN     NaN     NaN   17.00   12.00];
ICRP.fQco.pancreas =                           [NaN    NaN    NaN    NaN     NaN     NaN    1.00    1.00]; % not used in PBPK model
ICRP.fQco.skeleton.total =                     [NaN    NaN    NaN    NaN     NaN     NaN    5.00    5.00];
ICRP.fQco.skeleton.red_marrow =                [NaN    NaN    NaN    NaN     NaN     NaN    3.00    3.00];
ICRP.fQco.skeleton.trabecular_bone =           [NaN    NaN    NaN    NaN     NaN     NaN    0.90    0.90];
ICRP.fQco.skeleton.cortical_bone =             [NaN    NaN    NaN    NaN     NaN     NaN    0.60    0.60];
ICRP.fQco.skeleton.others =                    [NaN    NaN    NaN    NaN     NaN     NaN    0.50    0.50];
ICRP.fQco.skin =                               [NaN    NaN    NaN    NaN     NaN     NaN    5.00    5.00];
ICRP.fQco.spleen =                             [NaN    NaN    NaN    NaN     NaN     NaN    3.00    3.00];
ICRP.fQco.thyroid =                            [NaN    NaN    NaN    NaN     NaN     NaN    1.50    1.50]; % not used in PBPK model
ICRP.fQco.lymph_nodes =                        [NaN    NaN    NaN    NaN     NaN     NaN    1.70    1.70]; % not used in PBPK model
ICRP.fQco.gonads =                             [NaN    NaN    NaN    NaN     NaN     NaN    0.05    0.02]; % not used in PBPK model
ICRP.fQco.adrenals =                           [NaN    NaN    NaN    NaN     NaN     NaN    0.30    0.30]; % not used in PBPK model
ICRP.fQco.urinary_bladder =                    [NaN    NaN    NaN    NaN     NaN     NaN    0.06    0.06]; % not used in PBPK model
ICRP.fQco.all_other_tissues =                  [NaN    NaN    NaN    NaN     NaN     NaN    1.39    1.92]; % not used in PBPK model


%%% ========================================================================================================
%%% Model: Tissue blood flow (fraction of cardiac output and absolut values)
%%%
%%% Unit : fraction and L/min
%%%
%%%
for a = 1:length(human.subtype)
    
    age = human.subtype{a};
    human.(age).fQco      = T.initialize.tissueDB.NaN;
    human.(age).fQco(adi) = ICRP.fQco.fat(a);
    human.(age).fQco(bon) = ICRP.fQco.skeleton.total(a);
    human.(age).fQco(bra) = ICRP.fQco.brain(a);
    human.(age).fQco(gut) = ICRP.fQco.small_intestine(a)+ICRP.fQco.large_intestine(a);
    human.(age).fQco(hea) = ICRP.fQco.coronary_tissue(a);
    human.(age).fQco(kid) = ICRP.fQco.kidney(a);
    human.(age).fQco(liv) = ICRP.fQco.liver(a);
    human.(age).fQco(mus) = ICRP.fQco.skeletal_muscle(a);
    human.(age).fQco(ski) = ICRP.fQco.skin(a);
    human.(age).fQco(spl) = ICRP.fQco.spleen(a);
    
    %%% change of units: [%] -> fraction
    human.(age).fQco      = human.(age).fQco/100;
    
    human.(age).co        = ICRP.co(a);
    human.(age).Q.blo     = human.(age).fQco * human.(age).co;
    
end;

%%% ASSUMPTION:
%%%
%%% For all children age classes: fQco values were estimated based on the
%%% approach presented in Abraham et al, Arch Toxicol, Vol 79 (2005), pp.
%%% 63-73, except when experimental data are available
%%%

children    = {'human0u', 'human1u', 'human5u' ,'human10u','human15m','human15f'};
refAgeClass = {'human35f','human35f','human35f','human35f','human35m','human35f'};

%%% used experimental data for children
%%%
%%% Brain:
%%% Blood flow data in [ml/min/100g] that were converted to [L/min/kg] from
%%% Chiron et al, 1992
Q_perKg_bra = [50 59 71 68 57 57]*10/1000;
%%%
%%% Kidneys:
%%% Assume that blood flow per kg kidney tissue is independent of age
%%% according to Grunert et al, 1990

for a = 1:length(children)
    
    %%% target age (age) and age class used for scaling (refAge)
    age        = children{a};
    refAge     = refAgeClass{a};
    
    %%% assign volume and blood flow to rest of body (rob)
    refQ_blo       = human.(refAge).Q.blo;
    refQ_blo(rob)  = human.(refAge).co - sum(refQ_blo(refModel.T.intoVen));
    
    refOW_exp      = human.(refAge).OW.exp;
    refOW_exp(rob) = human.(refAge).BW - sum(human.(refAge).OW.exp(refModel.T.allTisExBlo))- ...
        human.(refAge).V.blood;
    
    tarOW_exp      = human.(age).OW.exp;
    tarOW_exp(rob) = human.(age).BW - sum(human.(age).OW.exp(refModel.T.allTisExBlo))- ...
        human.(age).V.blood;
    
    %%% intermediate target regional blood flow, scaled solely
    %%% according to ratio of target-to-reference tissue volumes
    SF_Q_blo   = tarOW_exp ./ refOW_exp;
    Q_inter    = SF_Q_blo .* refQ_blo;
    
    %%% use experimental data for children (see above)
    Q_blo      = T.initialize.tissueDB.NaN;
    Q_blo(kid) = Q_inter(kid);
    Q_blo(bra) = Q_perKg_bra(a)*human.(age).V.tis(bra);
    
    %%% Scale all regional tissue blood flows to match cardiac output.
    %%% First, scale all blood flows that flow into the vein,
    %%% including rest of body, but excluding those tissues where
    %%% experimental data were used (brain and kidney).
    tis = [refModel.T.intoVen rob];
    tis(kid==tis) = []; tis(bra==tis) = [];
    
    SF_co = (human.(age).co - Q_blo(kid) - Q_blo(bra)) / sum(Q_inter(tis));
    
    if SF_co < 1
        fprintf('\n Scaling blood flows to children resulted in cardiac output that is larger than the experimental reported one!\n\n');
    end;
    Q_blo(tis) = SF_co * Q_inter(tis);
    
    %Q_blo([art lun ven]) = human.(age).co - Q_blo(rob);
    
    %%% Second, scale all regional tissue blood flows that flow into the
    %%% liver (here gut and spleen). To this end, determine hepartic artery
    %%% blood flow (hepart)
    Q_hepart       = human.(refAge).Q.blo(liv)-human.(refAge).Q.blo(gut)-human.(refAge).Q.blo(spl);
    Q_perKg_hepart = Q_hepart / human.(refAge).V.tis(liv);
    Q_inter_hepart = Q_perKg_hepart*human.(age).V.tis(liv);
    
    SF_hepart  = Q_blo(liv) / (Q_inter_hepart + Q_inter(gut) + Q_inter(spl));
    if SF_hepart <1
        fprintf('\n Scaling blood flows to children resulted in liver blood flow that is larger than the experimental reported one!\n\n');
    end;
    tis = [gut spl];
    Q_blo(tis) = SF_hepart * Q_inter(tis);
    
    %%% assign regional blood flows and fractions of cardiac output
    human.(age).Q.blo = Q_blo;
    human.(age).fQco  = human.(age).Q.blo/human.(age).co;
    
end;


%%% ========================================================================================================
%%% Data  : Total tissue water (Vwt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: Poulin & Theil, J Pharm Sci, Vol 91, 2002 and Poulin & Theil, J Pharm Sci, 2009 for ery value
%%%
data.human.fVexp.wtot = T.initialize.tissueDB.NaN;
data.human.fVexp.wtot(lun) = 0.811;
data.human.fVexp.wtot(kid) = 0.783;
data.human.fVexp.wtot(spl) = 0.778;
data.human.fVexp.wtot(bra) = 0.77;
data.human.fVexp.wtot(mus) = 0.76;
data.human.fVexp.wtot(hea) = 0.758;
data.human.fVexp.wtot(liv) = 0.751;
data.human.fVexp.wtot(gut) = 0.718;
data.human.fVexp.wtot(ski) = 0.718;
data.human.fVexp.wtot(bon) = 0.439;
data.human.fVexp.wtot(adi) = 0.18;
data.human.fVexp.wtot(pla) = 0.945;
data.human.fVexp.wtot(ery) = 0.63;

%%% ASSUMPTION:
%%%
%%% Fractions of experimental tissue volumes are identical to the
%%% fractions of tissue volume
%%%
data.human.fVtis.wtot = data.human.fVexp.wtot;

%%% ASSUMPTION:
%%%
%%% Assume that ratio wex-to-wtot is the same as in RAT
%%%
r_wex_to_wtot   = data.rat.fVtis.wex ./ (data.rat.fVtis.wex+data.rat.fVtis.wic);
%%%
%%% fraction of extra-celluar water (wex)
data.human.fVtis.wex = r_wex_to_wtot .* data.human.fVtis.wtot;
data.human.fVtis.wex(ery) = 0;
%%%
%%% fraction of intracellular water (wic)
data.human.fVtis.wic = data.human.fVtis.wtot - data.human.fVtis.wex;


%%% ========================================================================================================
%%% Data  : Fraction of interstitial and intra-cellular space
%%%
%%% Unit  : fraction
%%%
%%% ASSUMPTION: values for human are identical to rat values
%%%
data.human.fVtis.int = data.rat.fVtis.int;
data.human.fVtis.cel = data.rat.fVtis.cel;



%%% ========================================================================================================
%%% Data  : Neutral lipids (fVtis.nlt) and neutal phospholipids (fVtis.npt)
%%%
%%% Unit  : fraction of tissue volume
%%% Source: Poulin and Theil, J Pharm Sci, Vol. 91, 2002
%%%         Rodgers and Rowland, J Pharm Res, Vol. 24, 2007 (erythrocyte
%%%         values only), (Table VII)
%%%
data.human.fVexp.nlt = T.initialize.tissueDB.NaN;  %%% neutral lipids
data.human.fVexp.nlt(adi) = 0.79;
data.human.fVexp.nlt(bon) = 0.074;
data.human.fVexp.nlt(bra) = 0.051;
data.human.fVexp.nlt(gut) = 0.0487;
data.human.fVexp.nlt(hea) = 0.0115;
data.human.fVexp.nlt(kid) = 0.0207;
data.human.fVexp.nlt(liv) = 0.0348;
data.human.fVexp.nlt(lun) = 0.003;
data.human.fVexp.nlt(mus) = 0.0238;
data.human.fVexp.nlt(ski) = 0.0284;
data.human.fVexp.nlt(spl) = 0.0201;
data.human.fVexp.nlt(pla) = 0.0035;
data.human.fVexp.nlt(ery) = 0.0033;
%%%
data.human.fVexp.npt = T.initialize.tissueDB.NaN;  %%% neutral phospholipids
data.human.fVexp.npt(bra) = 0.0565;
data.human.fVexp.npt(liv) = 0.0252;
data.human.fVexp.npt(spl) = 0.0198;
data.human.fVexp.npt(hea) = 0.0166;
data.human.fVexp.npt(gut) = 0.0163;
data.human.fVexp.npt(kid) = 0.0162;
data.human.fVexp.npt(ski) = 0.0111;
data.human.fVexp.npt(lun) = 0.009;
data.human.fVexp.npt(mus) = 0.0072;
data.human.fVexp.npt(adi) = 0.002;
data.human.fVexp.npt(bon) = 0.0011;
data.human.fVexp.npt(pla) = 0.00225;
data.human.fVexp.npt(ery) = 0.0012;

%%% ASSUMPTION:
%%%
%%% Fractions of experimental tissue volumes are identical to the fractions
%%% of tissue volume
%%%
data.human.fVtis.nlt = data.human.fVexp.nlt;
data.human.fVtis.npt = data.human.fVexp.npt;

%%% ========================================================================================================
%%% Data  : Intra-cellular acidic phospholipids (APmt)
%%%
%%% Unit  : fraction

%%% ASSUMPTION:
%%%
%%% Human values are identical to rat values
%%%
data.human.fVtis.APmt = data.rat.fVtis.APmt;


%%% ========================================================================================================
%%% Data  : Albumin (Atp)
%%%
%%% Unit  : Tissue-to-plasma ratio

%%% ASSUMPTION:
%%%
%%% Human values are identical to rat values
%%%
data.human.r.Atp = data.rat.r.Atp;


%%% ========================================================================================================
%%% Data  : Lipoprotein (Ltp)
%%%
%%% Unit  : Tissue-to-plasma ratio

%%% ASSUMPTION:
%%%
%%% Human values are identical to rat values
%%%
data.human.r.Ltp = data.rat.r.Ltp;


%%% ========================================================================================================
%%% Model : Assignment of fractional volumes and ratio to all age classes
%%%
for a = 1:length(human.subtype)
    
    subtype = human.subtype{a};
    human.(subtype).r     = data.human.r;
    human.(subtype).fVtis = data.human.fVtis;
    
end;

%%% set male/female children to corresponding unisex values
for subtype = {'human0','human1','human5','human10'}
    
    age_root   = char(subtype);
    age_uni    = [age_root,'u'];
    age_male   = [age_root,'m'];
    age_female = [age_root,'f'];
    
    human.(age_male)   = human.(age_uni);
    human.(age_female) = human.(age_uni);
    
end;


%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++ HUMAN +++
%%% ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



%%% ========================================================================================================
%%% Model : Assignment of species and tissue
%%%
species.type = {'human','rat','mouse'};

species.info  = info;

species.T     = T;

species.rat   = rat;
species.mouse = mouse;
species.human = human;


