%%% Version: June 26th, 2014
%%%
%%% This function specifies the graphical output related to the entire
%%% virtual population. Graphical output specific to some model should be
%%% coded in the corresponding model file.
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function [] = GenericPBPKmodel_graphicalOutput(Individual,ShowGraphics)


%%% set output units depending on drug class
%%%
ShowGraphics = GenericPBPKmodel_setOutputUnits(Individual,ShowGraphics);

%%% plot plasma concentration of entire population
%%%
if strcmp(ShowGraphics.plasma.allIndividuals,'yes')
    GenericPBPKmodel_plasmaConcentration(Individual,ShowGraphics)
end;

%%% plot concentration-time profiles of all tissues for specified
%%% Individuals
%%%
if ~isempty(ShowGraphics.allTissues.individual)
    GenericPBPKmodel_plotAllTissues(Individual,ShowGraphics)
end;

%%% plot 5th,25th, 50th, 75th and 95th percentile of concentration-time
%%% profiles of entire population (if larger than 9 Individuals)
%%%
if strcmp(ShowGraphics.percentiles.plasma,'yes') && (length(Individual)>=10)
    GenericPBPKmodel_precentiles(Individual,ShowGraphics)
end;

%%% plot total amount over time for debugging
%%%
if ~isempty(ShowGraphics.TestForLossOfMass.id)
    GenericPBPKmodel_checkForConservationOfAmounts(Individual,ShowGraphics)
end;

fprintf('\n');

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function ShowGraphics = GenericPBPKmodel_setOutputUnits(Individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% set default units depending on drug class
compound = Individual(1).drug.listOfDrugs{1};

switch Individual(1).drug.(compound).class
    
    case 'sMD'
        %%% scale time from [min] to [h]
        ShowGraphics.SF.time = 1/60;
        ShowGraphics.unit.time = 'h';
        
        %%% leave concentration in [mg/L]
        ShowGraphics.SF.conc = 1;
        ShowGraphics.unit.conc = 'mg/L';
        
    case 'mAb'
        %%% scale time from [min] to [day]
        ShowGraphics.SF.time = 1/(60*24);
        ShowGraphics.unit.time = 'day';
        
        %%% leave concentration in [mg]/L
        ShowGraphics.SF.conc = 1;
        ShowGraphics.unit.conc = 'mg/L';
        
        %%% uncomment below in case you want [nmol/L]
        ShowGraphics.SF.conc = Individual(1).drug.(compound).SF.mg_to_nmol;
        ShowGraphics.unit.conc = 'nmol/L';
        
    case 'pAb' %e.g. IVIG
        %%% scale time from [min] to [day]
        ShowGraphics.SF.time = 1/(60*24);
        ShowGraphics.unit.time = 'day';
        
        %%% leave concentration in [mg]/L
        ShowGraphics.SF.conc = 1;
        ShowGraphics.unit.conc = 'mg/L';
        
        %%% uncomment below in case you want [nmol/L]
        ShowGraphics.SF.conc = Individual(1).drug.(compound).SF.mg_to_nmol;
        ShowGraphics.unit.conc = 'nmol/L';
        
        
    otherwise
        
        %%% leave time in [min]
        ShowGraphics.SF.time = 1;
        ShowGraphics.unit.time = 'min';
        
        %%% leave concentration in [mg/L]
        ShowGraphics.SF.conc = 1;
        ShowGraphics.unit.conc = 'mg/L';
        
end;

ShowGraphics.upperLimitY = 10*ShowGraphics.lowerLimitY;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_plasmaConcentration(Individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% plot plasma concentrations for entire population

T = Individual(1).stdout.T; figNr = ShowGraphics.plasma.figNr;
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

%%% default limits of y axis
%%%
ymin = ShowGraphics.lowerLimitY;
ymax = ShowGraphics.upperLimitY;

figure(figNr); clf; hold on;
for id=1:length(Individual)
    
    t = Individual(id).stdout.t*SF.time;
    C_pla = (Individual(id).stdout.C.pla*SF.conc);
    
    
    color = char(Individual(id).color);
    
    if ~strcmp(Individual(id).model(1).type,'ExperimentalData')
        
        plot(t,C_pla,color,'LineWidth',1.3);
        xlim([t(1), t(end)]); ylim([ymin ymax]);
        
    else
        
        ind = find(isfinite(C_pla));
        plot(t(ind),C_pla(ind),'o','MarkerEdgeColor','k','MarkerFaceColor',color,'MarkerSize',7);
        xlim(Individual(id).stdout.limits.t(T.pla,:)*SF.time),ylim([ymin ymax]);
        
        
        %%% pred.errorbar in [nM]
        pred=Individual(id).pred;
        errorbar_up   =  pred.errorbar.up.C.pla;
        errorbar_down =  pred.errorbar.down.C.pla;
        errorbar(t(ind),C_pla(ind),errorbar_down(ind),errorbar_up(ind),'ko');
    end;
    
    if ~isnan(max(C_pla))
        ymax = max([ymax, max(C_pla)]);
    end;
    
end;

hold off; grid off;

title(sprintf('drug %s', Individual(1).stdout.compound));
xlabel(sprintf('t[%s]',unit.time));
ylabel(sprintf(' %s Conc [%s] in venous plasma',Individual(1).stdout.compound, unit.conc));

ylim([ymin 1.1*ymax]);

box on

set(gca,'YScale',ShowGraphics.plasma.YScale);
fprintf('\n');

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_plotAllTissues(Individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Give detailed plots for all tissues for specified Individuals -----

T = Individual(1).stdout.T; figNr = ShowGraphics.allTissues.figNr;
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

if sum(ismember(ShowGraphics.allTissues.individual,1:length(Individual))==0)
    GenericPBPKmodel_ReportErrorMessage('ShowGraphics.allTissues.individual contains invalid ids!');
end;

%%% default limits of y axis
%%%
ymin = ShowGraphics.lowerLimitY;
ymax = ShowGraphics.upperLimitY;

figure(figNr); clf;
for id = ShowGraphics.allTissues.individual
    
    hold on;
    t = Individual(id).stdout.t*SF.time;
    C = Individual(id).stdout.C.tis*SF.conc;
    color = char(Individual(id).color);
    
    %%% determine max concentration value for all non NaN concentration
    %%% time profiles
    %%%
    nonNaNtissues = find(isfinite(max(C)));
    if ~isempty(nonNaNtissues)
        ymax = max(ymax,max(max(C(:,nonNaNtissues))));  % take max of previous ymax and current one
    end;
    
    %%% adjust ymin, if too small
    %%%
    if ymax<=ymin
        fprintf('\n'); fprintf(2,'>>> ShowGraphics.lowerLimitY lowered, since it was smaller than Cmax! <<< \n');
        ymin = 1e-3*ymax;
    end;
    
    
        if strcmp(Individual(id).model(1).type,'mAb_PBPK_11CMT_extravasLim_int_Vgrowth_noTar_V')
     tissues = [T.adi T.bon T.bra T.gut T.hea T.kid T.liv T.lun T.mus T.ski T.spl T.tum];
        else
     tissues = [T.adi T.bon T.bra T.gut T.hea T.kid T.liv T.lun T.mus T.ski T.spl ];
        end
    for k = 1:length(tissues)
        
        subplot(3,4,k); hold on;
        tis = tissues(k);
        
        if ~strcmp(Individual(id).model(1).type,'ExperimentalData')
            
            plot(t,C(:,tis),color,'LineWidth',1.1);
            xlim([t(1), t(end)]); ylim([ymin ymax]);
        else
            ind = find(isfinite(C(:,tis)));
            plot(t(ind),C(ind,tis),'o','MarkerEdgeColor','k','MarkerFaceColor',color,'MarkerSize',7)
            xlim(Individual(id).stdout.limits.t(tis,:)*SF.time), ylim([ymin ymax]);
            
            %%%% errorbars in tissues: errorbars in [nM]
            pred=Individual(id).pred;
            errorbar_up   =  pred.errorbar.up.C.tis(:,tis);
            errorbar_down =  pred.errorbar.down.C.tis(:,tis);
            errorbar(t(ind),C(ind,tis),errorbar_down(ind),errorbar_up(ind),'ko');
        end;
        xlabel(sprintf('t[%s]',unit.time));
        
        ylabel(sprintf(' %s Conc [%s] in %s',Individual(1).stdout.compound, unit.conc, T.name{tis}));
        ylim([ymin 1.1*ymax]);
        set(gca,'YScale',ShowGraphics.allTissues.YScale);
        
    end;
    
    subplot(3,4,2);
    
    title(sprintf('drug %s', Individual(1).stdout.compound));
    
    set(gca,'YScale',ShowGraphics.allTissues.YScale);
    
    if strcmp(ShowGraphics.allTissues.oneFigOnly,'no') && id~=ShowGraphics.allTissues.individual(end)
        hold off;
        figNr = figNr + 1;
        figure(figNr); clf;
    end;
    
    
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_precentiles(Individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Plot percentiles for venous blood concentrations for entire population
%%% Only applicable for population of size>=10

T = Individual(1).stdout.T;  figNr = ShowGraphics.percentiles.figNr;
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

%%% check, whether all Individuals have predictions at identical time
%%% points
timepoints = Individual(1).pred.t;
for k = 2:length(Individual)
    if length(Individual(k).pred.t)~= length(timepoints) || sum(Individual(k).pred.t ~= timepoints)
        beep;
        fprintf(2,'   For the percentile plot to work you have to enforce identical\n');
        fprintf(2,'   time vectors for all Individuals.  \n\n');
        fprintf(2,'   Suggestion: Set pred.noOfTimePoints = 100; \n\n');
        return;
        
    end;
end;

%%% determine sex of Individual for coloring purpose
%%%
BW = NaN*ones(1,length(Individual)); BH = NaN*ones(1,length(Individual));
ind_male = []; ind_female = []; ind_uni = [];

for id=1:length(Individual)
    BW(id) = Individual(id).species.BW; BH(id) = Individual(id).species.BH;
    switch Individual(id).species.sex
        case 'male';
            ind_male = [ind_male, id];
        case 'female';
            ind_female = [ind_female, id];
    end;
end;

figure(figNr); clf;

%%% plot BW versus BH for all individulas
%%%
subplot(1,2,1);
plot(BW(ind_male),BH(ind_male),'ro',BW(ind_female),BH(ind_female),'bo',BW(ind_uni),BH(ind_uni),'ko');
xlabel('BW'); ylabel('BH');
fprintf('\n\n   Legend Fig.%d. Left: male (r), female (b). Right: 5th, 25th. 50th, 75th and 95th percentiles\n\n',ShowGraphics.percentiles.figNr);

tspan = Individual(1).stdout.t*SF.time;
if length(tspan)<5
    fprintf(' Only %d time points specified in study.observationTime! Please increase number for more meaningful graphical representation!');
    beep;
end;

%%% combine all plasma concentration-time profiles in a single matrix
%%%

ind = 1:length(tspan);

C_pla = NaN*ones(length(Individual),length(Individual(1).stdout.t(ind)));
for id=1:length(Individual)
    C_pla(id,:) = Individual(id).stdout.C.pla(ind,:)';
end;

subplot(1,2,2);
percentile = prctile(C_pla,[5 25 50 75 95])';

hold on;
h = area(tspan(ind),[percentile(:,1)'; diff(percentile,1,2)']');
set(get(h(1),'Children'),'FaceColor',[1 1 1],'LineStyle','none');
set(get(h(2),'Children'),'FaceColor',[.9 .9 .9],'LineStyle','none');
set(get(h(3),'Children'),'FaceColor',[.8 .8 .8],'LineStyle','none');
set(get(h(4),'Children'),'FaceColor',[.8 .8 .8],'LineStyle','none');
set(get(h(5),'Children'),'FaceColor',[.9 .9 .9],'LineStyle','none');

plot(tspan(ind),percentile(:,3),'r-',tspan(ind),percentile(:,[1 5]),'k-',...
    tspan(ind),percentile(:,[2 4]),'k--','LineWidth',1.3);
set(gca,'Layer','top');

title(sprintf('Compound = %s',Individual(id).drug.listOfDrugs{1}));
xlabel(sprintf('t[%s]',unit.time));
ylabel(sprintf('C [%s] in venous plasma',unit.conc));
xlim([tspan(1), tspan(end)]);

ymin = ShowGraphics.lowerLimitY;

ymax = max(max(C_pla));
if ymax<=ymin
    fprintf('\n'); fprintf(2,'>>> ShowGraphics.lowerLimitY lowered, since it was smaller than Cmax! <<< \n');
    ymin = 1e-3*ymax;
end;
grid on; ylim([ymin ymax]);
ShowGraphics.percentiles.YScale
set(gca,'YScale',ShowGraphics.percentiles.YScale);


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_checkForConservationOfAmounts(Individual,ShowGraphics)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% Give detailed plots for amounts (total, metabolized and sum) for specified Individuals -----

T = Individual(1).stdout.T; figNr = ShowGraphics.TestForLossOfMass.figNr;
SF = ShowGraphics.SF; unit = ShowGraphics.unit;

if sum(ismember(ShowGraphics.TestForLossOfMass.id,1:length(Individual))==0)
    GenericPBPKmodel_ReportErrorMessage('ShowGraphics.TestForLossOfMass.id contains invalid ids!');
end;

experimental_ids = [];
for k=1:length(ShowGraphics.TestForLossOfMass.id)
    
    id = ShowGraphics.TestForLossOfMass.id(k);
    if strcmp(Individual(id).model.type,'ExperimentalData')
        experimental_ids = [experimental_ids id];
    end;
    
end;
if ~isempty(experimental_ids)
    ShowGraphics.TestForLossOfMass.id = setdiff(ShowGraphics.TestForLossOfMass.id,experimental_ids);
    fprintf('\n\n   Experimental data excluded from test for loss of mass! \n\n');
end;


figure(figNr); clf; hold on;
for id = ShowGraphics.TestForLossOfMass.id
    
    t = Individual(id).stdout.t*SF.time;
    A.body    = Individual(id).stdout.A.body;
    A.metab   = Individual(id).stdout.A.metab;
    A.GItract = Individual(id).stdout.A.GItract;
    A.IVbag   = Individual(id).stdout.A.IVbag;
    A.all     = A.body + A.metab + A.GItract + A.IVbag;
    color = char(Individual(id).color);
    
    plot(t,A.all/A.all(1),color,'LineWidth',1.3);
    xlabel(sprintf('t[%s]',unit.time));
    ylabel('rel change in (A_{body}+A_{metab}+A_{GItract}+A_{IVbag})');
    xlim([t(1), t(end)]); ylim([0.99 1.01]);
    set(gca,'YScale','lin');
    
    title('If you do not see a straight line, check your model equations!');
    
end;

end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



