%%% Version: November 6th, 2014
%%%
%%%
%%% This function specifies a simplfied PBPK model for mAb with
%%% extravasation-limited tissue distribution, including ENDOGENOUS IgG
%%% but no target, parameterized in terms of interstitial volumes, with non-linear
%%% clearance from plasma (FcRn dependent clearance); Cutoff Model for
%%% definition of fraction unbound to FcRn with FcRn-IgG interaction kinetically linked to plasma

%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

function Individual = GenericPBPKmodel_mAb_PBPK_11CMT_extravasLim_int_IgGendo(Individual)

%%% check whether drug class is really sMD
%%%
GenericPBPKmodel_checkForModelApplicability(Individual);

%%% define model specific parameters
%%%
Individual = GenericPBPKmodel_defineModelParameters(Individual);

%%% simulate PBPK model
%%%
model = Individual.model; dosingEvent = Individual.pred.dosingEvent;
X0 = model.X0; sim.t = []; sim.X = [];

for d=1:length(dosingEvent)
    
    %%% administration details and simulation time span
    admin = dosingEvent(d).admin;
    tspan = dosingEvent(d).tspan;
    
    %%% account for dosing
    X0 = GenericPBPKmodel_drugAdministration(X0,model,admin);
    
    %%% solve system of ODEs 'GenericPBPKmodel_RHS' with initial conditions 'X0'
    [t,X] = ode15s(@GenericPBPKmodel_RHS,tspan,X0',[],Individual);
    
    %%% store current output in sim structure
    sim.t = [sim.t; t]; sim.X = [sim.X; X];
    X0 = X(end,:)';
    
end;

%%% determine standard simulation output
%%%
Individual = GenericPBPKmodel_determineStandartOutput(Individual,sim);

%%% normalized concentration plot
%%%
GenericPBPKmodel_normalizedConcentrationPlot(Individual);

%%% graphical output specific to this model
%%%
GenericPBPKmodel_specificGraphicalOutput(Individual);

end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_checkForModelApplicability(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

numberOfDrugs = length(Individual.drug.listOfDrugs);
if ~ismember(numberOfDrugs,[1,2])
    message = sprintf('Number of drugs (k=%d) not compatable with model >>%s<<',...
        numberOfDrugs,Individual.model.type);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

listOfAllowedCompounds = {'mAb','IVIG'};
for k = 1:numberOfDrugs
    
    compound = Individual.drug.listOfDrugs{k};
    drug     = Individual.drug.(compound);
    
    if ismember(drug.class,listOfAllowedCompounds)
        listOfAllowedCompounds = setdiff(listOfAllowedCompounds,drug.class);
    elseif ismember(drug.name,listOfAllowedCompounds)
        listOfAllowedCompounds = setdiff(listOfAllowedCompounds,drug.name);
    else
        message = sprintf('Drugs choosen for dosing not compatible with model >>%s<<',...
            Individual.model.type,Individual.drug.(compound).class);
        GenericPBPKmodel_ReportErrorMessage(message);
        
    end;
    
end;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_defineModelParameters(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model = Individual.model;

%%% -----------------------------------------------------------------------
%%% Define indexing
%%%

%%% tissue topology indexing
%%%
S = Individual.species.T;

S.allCmts  = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv S.pla];
S.organs   = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski S.gut S.spl S.liv      ];
S.plasma   = [                                                            S.pla];
S.tissueDB = S.allCmts; % specify which tissues are part of the general tissue DB

S.visOrg      = [S.gut S.spl S.liv];
S.nonvisOrg   = [S.lun S.adi S.hea S.kid S.mus S.bon S.ski];

S.maxIndex.org = max(S.allCmts);
S.initialize.org.NaN   = NaN*ones(1,S.maxIndex.org);

%%% compartment indexing
%%%
S.C_pla = 1;                      % total plasma compartment
S.bolus = S.C_pla;                % i.v. bolus administration
S.C_int = (1:S.maxIndex.org) + 1; % interstitial compartments

S.maxIndex.cmt = max(S.C_int);

%%% dosing and metabolism indexing
%%%
S.GItract = 1 + S.maxIndex.cmt; % gastro-intestinal tract (for po dosing)
S.IVbag   = 2 + S.maxIndex.cmt; % IVbag (for infusion dosing)
S.IVrate  = 3 + S.maxIndex.cmt; % IVbag (for infusion dosing)
S.metab   = 4 + S.maxIndex.cmt; % metabolized drug

S.maxIndex.dosing = 4 + S.maxIndex.cmt;

%%% additional indexing
%%%
S.maxIndex.each = 0 + S.maxIndex.dosing;

%%% drug indexing
S.mAb = [1:S.maxIndex.each];
S.IgG = [1:S.maxIndex.each] + S.maxIndex.each;

%%% define range of indexes also based on drug name
%%%
for k = 1:length(Individual.drug.listOfDrugs)
    
    compound = Individual.drug.listOfDrugs{k};
    drug     = Individual.drug.(compound);
    if strcmp(drug.class,'mAb')
        S.(compound) = S.mAb;
    elseif strcmp(drug.name,'IVIG')
        S.(compound) = S.IgG;
    end;
    
end;

%%% additional indexing
%%%
S.maxIndex.all = 2*S.maxIndex.each;


%%% -----------------------------------------------------------------------
%%% Define PBPK parameters
%%%

org = S.organs; pla = S.pla;
species  = Individual.species;

compound = Individual.drug.listOfDrugs{1};
mAb      = Individual.drug.(compound);

%%% tissue volumes
%%%
V.tis          = S.initialize.org.NaN;
V.tis(org)     = species.V.tis(org);

fVtis.int      = S.initialize.org.NaN;
fVtis.int(org) = species.fVtis.int(org);

V.int          = S.initialize.org.NaN;
V.int(org)     = fVtis.int(org) .* V.tis(org);

%%% total plasma volume
V.pla = (1-species.hct) * species.V.blood;

%%% volume for i.v. bolus dosing
%%%
V.iv_bolus = V.pla;

%%% blood flows (ensure closed circulatory system!)
%%%
Q.blo = S.initialize.org.NaN;
Q.blo(org)    = species.Q.blo(org);
Q.blo(S.lun)  = species.co;

%%% peripheral plasma flows
Q.pla = (1-species.hct) * Q.blo;

%%% lymph flow as fraction of peripheral plasma and blood flow
%%%
fLymph = S.initialize.org.NaN;
fLymph(S.visOrg)    = 0.02;
fLymph(S.nonvisOrg) = 0.04;

Q.lymph = fLymph.*Q.pla;

Q.lymph(pla) = sum(Q.lymph(org));

%%% define vascular reflection coefficients
%%%
sigma.vas = S.initialize.org.NaN;

%%%sigmas Brown Mouse

sigma.vas([S.liv S.spl S.gut])            = 0.95;
sigma.vas([S.hea S.ski S.kid S.lun])      = 0.975;
sigma.vas([S.mus S.adi S.bon])            = 0.99;

sigma.pla = 1 - sum( Q.lymph(org).*(1-sigma.vas(org)) ) /Q.lymph(pla);


%%% drug specific parameters starting below
%%%

ABC.tis      = S.initialize.org.NaN;
ABC.tis(org) = mAb.ABC.tis(org);
ABC.pla      = mAb.ABC.tis(pla);

ABC.pla      = mAb.ABC.tis(pla);

%%% define elimination-corrected tissue partition coefficients
eK.tis       = S.initialize.org.NaN;
eK.tis(org)  = ABC.tis(org)./(1-sigma.vas(org));
eK.pla       = ABC.pla./(1-sigma.pla);

%%% blood-to-plasma ratio assuming that mAb do not distribute or bind to
%%% erythrocytes
%%%
BP = (1-species.hct);

%%% tissue-specific extraction ratio
E.tis      = S.initialize.org.NaN;
E.tis(org) = 0;

%%% define unbound plasma clearance and total FcRn concentration with CLpla = CLupla * fu

fu=1;
if isfield(model,'Estimation_plasmaData') && strcmp(model.Estimation_plasmaData,'yes')
    
    %%%final estimates for WTmice:
    if isfield(model,'WTmice') && strcmp(model.WTmice,'yes')
        %estimated with OFV 6.968253e-01
        CLupla.pla  = 7.946810e-07*1/fu;
        FcRn.max    = 1.316688e+04;
    end
    
    %%%estimates for nude mice:
    if isfield(model,'NudeMice') && strcmp(model.NudeMice,'yes')
        %estimated with OFV 1.938551e-01
        CLupla.pla =  2.659517e-07*1/fu; %estimated value for reference BW 25 g
        FcRn.max   =  1.065589e+00;
    end;
    
    %%%estimates for SCID mice:
    if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
        %estimated  with OFV  2.297227e+00
        CLupla.pla   = 2.564997e-07*1/fu;   %estimated value for reference BW 25 g
        FcRn.max     = 6.473919e-01;
        
    end;
end

%%% steady state endogenous IgG level in plasma
if isfield(model,'WTmice') && strcmp(model.WTmice,'yes')
    IgGendoSS_pla      = 1.47e4;             % in [nM] steady-state mouse IgG conc for WT mice
end;

if isfield(model,'SCIDmice') && strcmp(model.SCIDmice,'yes')
    IgGendoSS_pla      = 66e-02;    % in [nM] steady-state  IgG conc for SCID mice
end;

if isfield(model,'NudeMice') && strcmp(model.NudeMice,'yes')
    IgGendoSS_pla      = 3.3e2;    % in [nM] steady-state  IgG conc for nude mice
end;

%%% estimation mode
if strcmp(Individual.estimation.modus,'on')
    E.tis(org)     = 0;
    CLupla.pla     = Individual.estimation.CLupla_tot;
    FcRn.max       = Individual.estimation.MAX;
    
end;

%%%scale of CLupla
CLupla.pla =  CLupla.pla *(Individual.species.BW/0.025);


fu = (IgGendoSS_pla - FcRn.max)/ IgGendoSS_pla;

if fu<0
    message= sprintf('WARNING fu= %1.3e turns negative \n', fu);
    GenericPBPKmodel_ReportErrorMessage(message);
end;


%%% production rate of endogenous IgG
IgGproduction_rate =  CLupla.pla* fu* IgGendoSS_pla;   % in [nM/min]

if IgGproduction_rate < 0
    message= sprintf('WARNING IgGproduction_rate = %1.3e turns negative\n',IgGproduction_rate);
    GenericPBPKmodel_ReportErrorMessage(message);
end;

%%% partition coefficients
%%%
K.tis(org) = eK.tis(org)./(1-E.tis(org));
K.pla      = 1;

%%% define intrinsic tissue clearance
%%%
CLint.tis = S.initialize.org.NaN;
CLint.tis(org) = Q.lymph(org)./K.tis(org).*E.tis(org)./(1-E.tis(org));

%%% account for FcRn status
if isfield(model,'FcRn_status') && strcmp(model.FcRn_status,'knock-out')
    
    FcRn.max = 0;
    
    %%% assuming that the endogenous IgG production rate is not effected,
    %%% the steady state of endogenous IgG in ko mice is
    
    fu=1;
    IgGendoSS_pla = IgGproduction_rate/CLupla.pla*fu;
    
end;

%%% compute ABCs values and effective flows
Leff.lymph   = (1-sigma.vas).*Q.lymph;

%%% compute ABCs, eK and K with respect to interstitial volume
%%% based on volume scaling factors
%%%
SF = mAb.SF;
SF.int2tis = species.fVtis.int;
SF.tis2int = 1./SF.int2tis;

ABC.int      = S.initialize.org.NaN;
ABC.int(org) = SF.tis2int(org).*ABC.tis(org);

eK.int       = S.initialize.org.NaN;
eK.int(org)  = SF.tis2int(org).*eK.tis(org);

K.int        = S.initialize.org.NaN;
K.int(org)   = SF.tis2int(org).*K.tis(org);

E.int        = S.initialize.org.NaN;
E.int        = E.tis; % note: scaling factors cancel out!

CLint.int    = S.initialize.org.NaN;
CLint.int    = Q.lymph./K.int.*E.int./(1-E.int);

%%% initial condition of ODEs
%%% set steady state concentrations for endogenous IgG
IgG.X0               = zeros(size(S.IgG));
IgG.X0(S.C_pla)      = IgGendoSS_pla;
IgG.X0(S.C_int(org)) = ABC.int(org)*IgGendoSS_pla;

X0        = zeros(1,S.maxIndex.all);
X0(S.IgG) = IgG.X0;


%%% -----------------------------------------------------------------------
%%% Assign model parameters
%%%

model.S      = S;
model.V      = V;
model.Q      = Q;
model.K      = K;
model.E      = E;
model.eK     = eK;
model.BP     = BP;
model.fLymph = fLymph;
model.sigma  = sigma;
model.CLint  = CLint;
model.CLupla = CLupla;
model.FcRn   = FcRn;
model.IgGproduction_rate = IgGproduction_rate;
model.ABC    = ABC;
model.L      = Leff;
model.SF     = SF;
model.X0     = X0;

Individual.model = model;

%%% -----------------------------------------------------------------------
%%% double check, whether there is some parameter values resulting in
%%% NaN values in the ODE
%%%
dX = GenericPBPKmodel_RHS(0,Individual.model.X0',Individual);
if sum(isnan(dX))
    message = sprintf('Some parameter values of id=%d cause the RHS of the ODE to have NaN values!',Individual.id);
    GenericPBPKmodel_ReportErrorMessage(message);
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [X0] = GenericPBPKmodel_drugAdministration(X0,model,admin)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% time and dose are expected to be in internal units [min] and [nmol]

S = model.S;

for k=1:length(admin)
    
    dose     = admin(k).dose;
    compound = admin(k).drug;
    
    %%% part of vector X0_cmp corresponds to the compound being dosed
    X0_cmp = X0(S.(compound));
    
    switch admin(k).route
        
        
        
        case 'iv_bolus';
            X0_cmp(S.bolus) = X0_cmp(S.bolus) + dose/model.V.iv_bolus;
            
        case 'iv_infusion';
            if dose>0 % start of iv infusion
                X0_cmp(S.IVbag)  = X0_cmp(S.IVbag) + dose;
                X0_cmp(S.IVrate) = X0_cmp(S.IVrate) + admin(k).infusionRate;
            else      % end of iv infusion
                X0_cmp(S.IVrate) = X0_cmp(S.IVrate) - admin(k).infusionRate;
            end;
            
        case 'po';
            %%% double check that p.o. dose = 0
            message = 'po administration of mAb/pAb not supported!';
            GenericPBPKmodel_ReportErrorMessage(message);
            
    end;
    
    %%% re-assign the updated part to X0
    X0(S.(compound)) = X0_cmp;
    
end;

if sum(isnan(X0))
    message = sprintf('Some parameter values of id=%d cause the initial condition X0 of the ODE to have NaN values!',Individual.id);
    GenericPBPKmodel_ReportErrorMessage(message);
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function dX = GenericPBPKmodel_RHS(t,X,Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% initialize output vector
dX = zeros(size(X));

%%% model and indexing
model = Individual.model; S = model.S;

%%% variables (always use row vector notation)
mAb.X = X(S.mAb)'; IgG.X = X(S.IgG)';

%%% assign mAb and (endogenous) IgG concentrations
mAb.C_pla = mAb.X(S.C_pla); mAb.C_int = mAb.X(S.C_int);
IgG.C_pla = IgG.X(S.C_pla); IgG.C_int = IgG.X(S.C_int);

%%% tissue volumes, blood flows, endosomal clearance etc.
V_pla    = model.V.pla;
V_int    = model.V.int;
Q        = model.Q.lymph;
K        = model.K.int;
sigma    = model.sigma.vas;
CLupla   = model.CLupla.pla;
FcRn     = model.FcRn.max;
IgG.production_rate = model.IgGproduction_rate;


%%% determine plasma clearance based on total IgG concentration
if (mAb.C_pla+IgG.C_pla)>FcRn
    fu = 1 - FcRn/( mAb.C_pla + IgG.C_pla );
else
    fu = 0;
end;
CLpla = CLupla * fu;

%%% infusion
mAb.infusion_rate = mAb.X(S.IVrate);
IgG.infusion_rate = IgG.X(S.IVrate);


%%% -----------------------------------------------------------------------
%%% START OF ODEs
%%%
mAb.VdC_int = zeros(size(mAb.C_int)); IgG.VdC_int = zeros(size(IgG.C_int)); org = S.organs;

%%% interstitial spaces
mAb.VdC_int(org) = Q(org).*( (1-sigma(org))*mAb.C_pla - mAb.C_int(org)./K(org) ); %no CLint because E(org) is zero
IgG.VdC_int(org) = Q(org).*( (1-sigma(org))*IgG.C_pla - IgG.C_int(org)./K(org) );

%%% plasma
mAb.VdC_pla = mAb.infusion_rate ...
    + sum( Q(org).*mAb.C_int(org)./K(org) ) ...
    - sum( Q(org).*(1-sigma(org))*mAb.C_pla ) - CLpla*mAb.C_pla;

IgG.VdC_pla = IgG.production_rate + IgG.infusion_rate ...
    + sum( Q(org).*IgG.C_int(org)./K(org) ) ...
    - sum( Q(org).*(1-sigma(org))*IgG.C_pla ) - CLpla*IgG.C_pla;

%%% drug amount in IVbag for infusion
mAb.dA_IVbag = -mAb.infusion_rate;
IgG.dA_IVbag = -IgG.infusion_rate;

%%% drug amount metabolized or excreted (discounted for IgG synthesis)
mAb.dA_metab = +CLpla*mAb.C_pla;
IgG.dA_metab = +CLpla*IgG.C_pla - IgG.production_rate;

%%%
%%% END OF ODEs
%%% -----------------------------------------------------------------------


%%% converting amounts to concentrations
mAb.dC_pla = mAb.VdC_pla./V_pla;
IgG.dC_pla = IgG.VdC_pla./V_pla;

mAb.dC_int = zeros(size(mAb.C_int)); IgG.dC_int = zeros(size(IgG.C_int));
mAb.dC_int(org) = mAb.VdC_int(org)./V_int(org);
IgG.dC_int(org) = IgG.VdC_int(org)./V_int(org);

%%% output vector (always in column vector notation)
mAb.dX = zeros(size(mAb.X)); IgG.dX = zeros(size(IgG.X));
mAb.dX(S.C_pla) = mAb.dC_pla';
mAb.dX(S.C_int) = mAb.dC_int';
mAb.dX(S.IVbag) = mAb.dA_IVbag';
mAb.dX(S.metab) = mAb.dA_metab';

IgG.dX(S.C_pla) = IgG.dC_pla';
IgG.dX(S.C_int) = IgG.dC_int';
IgG.dX(S.IVbag) = IgG.dA_IVbag';
IgG.dX(S.metab) = IgG.dA_metab';

dX(S.mAb) = mAb.dX;
dX(S.IgG) = IgG.dX;

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_determineStandartOutput(Individual,sim)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

model    = Individual.model;
pred     = Individual.pred;  info = Individual.info;

S = model.S;

%%% initialize all compartments with NaN and assign only to those
%%% compartments values that are part of the model topology
initialize.NaN = ones(size(sim.t))*S.initialize.org.NaN;
org = S.organs;

list_of_drugs = {'mAb','IgG'};

for k=1:length(list_of_drugs)
    
    compound = list_of_drugs{k};
    
    %%% -----------------------------------------------------------------------
    %%% assign predictions (in internal units)
    %%%
    info.pred.t         = 'simulation time in [min]';
    pred.(compound).t       = sim.t;
    
    sim.(compound).X     = sim.X(:,S.(compound));
    sim.(compound).X_pla = sim.(compound).X(:,S.C_pla);
    sim.(compound).X_int = sim.(compound).X(:,S.C_int);
    
    %%% plasma concentration
    %%%
    info.pred.(compound).C.pla     = sprintf('concentration of %s in plasma [nmol/L]',compound);
    pred.(compound).C.pla          = sim.(compound).X_pla;
    
    %%% interstitial concentrations
    %%%
    info.pred.(compound).C.int     = sprintf('concentration of %s in interstitial space [nmol/L]',compound);
    pred.(compound).C.int          = initialize.NaN;
    pred.(compound).C.int(:,org)   = sim.(compound).X_int(:,org);
    
    %%% tissue concentrations
    %%%
    info.pred.(compound).C.tis     = sprintf('concentration of %s in tissue space [nmol/L]',compound);
    pred.(compound).C.tis          = initialize.NaN;
    pred.(compound).C.tis(:,org)   = pred.(compound).C.int(:,org)*diag(model.SF.int2tis(org));
    
    %%% various amounts
    %%%
    info.pred.(compound).A.pla     = sprintf('amount of %s in plasma [nmol]',compound);
    pred.(compound).A.pla          = pred.(compound).C.pla*model.V.pla;
    
    info.pred.(compound).A.int     = sprintf('amount of %s in interstitial space [nmol]',compound);
    pred.(compound).A.int          = initialize.NaN;
    pred.(compound).A.int(:,org)   = pred.(compound).C.int(:,org)*diag(model.V.int(org));
    
    info.pred.(compound).A.tis     = sprintf('amount of %s in tissue space [nmol]',compound);
    pred.(compound).A.tis          = pred.(compound).A.int;
    
    info.pred.(compound).A.body    = sprintf('total amount of %s in the body in [nmol]',compound);
    pred.(compound).A.body         = sum(pred.(compound).A.int(:,org),2)+pred.(compound).A.pla;
    
    info.pred.(compound).A.GItract = sprintf('remaining amount of %s in the GI tract in [nmol]',compound);
    pred.(compound).A.GItract      = sim.(compound).X(:,S.GItract);
    
    info.pred.(compound).A.IVbag   = sprintf('remaining amount of %s in the IVbag in [nmol]',compound);
    pred.(compound).A.IVbag        = sim.(compound).X(:,S.IVbag);
    
    info.pred.(compound).IVrate    = sprintf('%s infusion rate in [nmol/min]',compound);
    pred.(compound).IVrate         = sim.(compound).X(:,S.IVrate);
    
    info.pred.(compound).A.metab   = sprintf('amount of %s metabolized in [nmol]',compound);
    pred.(compound).A.metab        = sim.(compound).X(:,S.metab);
    
end;

%%% fraction unbound of total IgG
mAb_pla = pred.mAb.C.pla; IgG_pla = pred.IgG.C.pla;
FcRn_max = model.FcRn.max;
IgG_tot = mAb_pla + IgG_pla;

fu = NaN*zeros(size(sim.t));
fu(IgG_tot<FcRn_max) = 0;
fu(IgG_tot>=FcRn_max) = 1 - FcRn_max./IgG_tot((IgG_tot>=FcRn_max));

pred.mAb.fu = fu;
pred.IgG.fu = fu;


%%% -----------------------------------------------------------------------
%%% determine standard output in [mg/L] or [mg]
%%%
stdout = Individual.stdout; stdout.T = Individual.species.T; T = stdout.T;
initialize.NaN = ones(size(sim.t))*T.initialize.tissueDB.NaN; SF = model.SF;

%%% which compound to report on in the standard output,
%%% either 'mAb' or 'IgG
%%%
if strcmp (Individual.stdout.compound, 'IgG');
    
    compound = 'IgG';
else
    compound = 'mAb';  %% default:%compound = 'mAb';
end

info.stdout.compound  = 'compound shown in standard output';
stdout.compound       = compound;

info.stdout.t         = 'simulation time in [min]';
stdout.t              = pred.(compound).t;

%%% tissue concentrations
%%%
info.stdout.C.tis     = 'concentration in tissue space [mg/L]';
stdout.C.tis          = initialize.NaN;
for k = intersect(T.tissueDB,S.organs)
    stdout.C.tis(:,k) = SF.nmol_to_mg*pred.(compound).C.tis(:,k);
end;

%%% plasma concentrations
%%%
info.stdout.C.pla     = 'concentration in plasma [mg/L]';
stdout.C.pla  = SF.nmol_to_mg*pred.(compound).C.pla;

%%% correct for residual blood
%%%
if isfield(stdout,'correct_for_residual_blood') && strcmp(stdout.correct_for_residual_blood,'yes')
    
    species = Individual.species;
    V.vas   = species.V.vas; V.tis = species.V.tis; hct = species.hct;
    
    V.vas   = species.V.res;
    
    for org = intersect(T.tissueDB,S.organs)
        stdout.C.tis(:,org) = ( V.tis(org).*stdout.C.tis(:,org) + (1-hct)*V.vas(org)*stdout.C.pla )./( V.vas(org)+V.tis(org) );
    end;
    
else
    stdout.correct_for_residual_blood = 'no';
end;


%%% various amounts
%%%
info.stdout.A.body    = 'total amount in the body in [mg]';
stdout.A.body         = SF.nmol_to_mg*pred.(compound).A.body;

info.stdout.A.GItract = 'remaining amount in the GI tract in [mg]';
stdout.A.GItract      = SF.nmol_to_mg*pred.(compound).A.GItract;

info.stdout.A.IVbag   = 'remaining amount in the IVbag in [mg]';
stdout.A.IVbag        = SF.nmol_to_mg*pred.(compound).A.IVbag;

info.stdout.IVrate    = 'infusion rate in [mg/min]';
stdout.IVrate         = SF.nmol_to_mg*pred.(compound).IVrate;

info.stdout.A.metab   = 'amount metabolized in [mg]';
stdout.A.metab        = SF.nmol_to_mg*pred.(compound).A.metab;


%%% final assignment
%%%
Individual.info   = info;
Individual.pred   = pred;
Individual.stdout = stdout;

end

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [] = GenericPBPKmodel_normalizedConcentrationPlot(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


if ~isfield(Individual.pred,'plot_normalized_conc') || ~strcmp(Individual.pred.plot_normalized_conc,'yes')
    return;
end;

compound = 'mAb';
pred = Individual.pred.(compound); model = Individual.model;

t = pred.t; S = Individual.model.S; id = Individual.id;

initialize.NaN = ones(size(pred.t))*S.initialize.org.NaN;
org = S.organs; pla = S.pla;

%%% define original and normalized concentrations
%%%
C  = initialize.NaN;
nC = initialize.NaN;

C(:,org)  = pred.C.int(:,org);
C(:,pla)  = pred.C.pla;
for k = org
    nC(:,k) = C(:,k) / model.ABC.int(k) ;
end;
nC(:,pla)= C(:,pla);


figure(100+id); clf
%%% plot all concentrations in a single plot
%%%
subplot(1,2,1); hold on;
for org = S.allCmts
    plot(t,C(:,org),S.style.(S.name{org}),'LineWidth',1.3);
end;
title(sprintf('Individual(%d), drug %s',id,Individual.drug.listOfDrugs{1}),'Fontsize',15);
xlabel('t[min]'); ylabel('C [nM]');
xlim([0, t(end)]); ylim([1e-3 1.5]*max(max(C)));
set(gca,'YScale','log'); hold off;


%%% plot all normalized concentrations in a single plot
%%%
subplot(1,2,2); hold on;
for org = S.allCmts
    plot(t,nC(:,org),S.style.(S.name{org}),'LineWidth',1.3);
end;
title(sprintf('Normalized drug concentration'),'Fontsize',15);
legend(S.name(S.tissueDB),'Location','SouthEast');
xlabel('t[min]'); ylabel('C [nM]');
xlim([0, t(end)]); ylim([1e-3 1.5]*max(max(nC)));
set(gca,'YScale','log'); hold off;

end



%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function Individual = GenericPBPKmodel_specificGraphicalOutput(Individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% define here specific output for this model

id = Individual.id;

plot_for_id = [id]; % example: plot_for_id = [1]    change in virtual population to id+1 therefore write here [id] instead of a number [2]

if ~ismember(id,plot_for_id)
    return;
end;

model    = Individual.model; S = model.S;
compound = 'mAb';
pred     = Individual.pred.(compound); t = pred.t/(60*24);


figure (200+id)
plot(t,pred.fu,'LineWidth',2);
xlabel('t [days]'); ylabel('fraction not bound to FcRn in endosomes');
ylim([-0.1 1.1]);




end

%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
