%%% Version: October 31st, 2014
%%%
%%% This function estimates the total clearance from plasma
%%%
%%% For references and citation, please see MAIN script.
%%%
%%% Copyright (C) 2014, Universitaet Potsdam, Germany
%%% Contact: W. Huisinga, huisinga@uni-potsdam.de
%%%
%%% The program is distributed under the terms of the
%%% Creative Commons License (CC BY-NC-SA 3.0):
%%% Attribution-NonCommercial-ShareAlike 3.0 Unported
%%%
%%% For a SHORT HUMAN-READABLE SUMMARY OF THE LEGAL CODE, see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/
%%%
%%% For the Legal Code (the full license) see URL
%%% http://creativecommons.org/licenses/by-nc-sa/3.0/legalcode
%%%
%%%


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


function [individual,estimation] = GenericPBPKmodel_estimatePBPKparameters_mAb_int(individual)


[individual,listOfIds] = GenericPBPKmodel_checkForEstimationModusOn(individual);

%%% if there is nothing to estimate --> return, otherwise report
%%%
if isempty(listOfIds.estimation)
    estimation = []; return;
else
    fprintf('\n   (*) Estimating PBPK parameters(s) ...\n');
end;

%%% load experimental data
%%%
for id=listOfIds.expData
    individual(id) = GenericPBPKmodel_ExperimentalData(individual(id));
end;

%%% determine time points for simulation to compare to experimental data
%%%
for id=listOfIds.estimation
    
    tpoints = [];
    for k = individual(id).estimation.expIds
        tpoints = union(tpoints, individual(k).pred.t);
    end;
    individual(id).pred.timeSpan = tpoints';
    individual(id).pred.timeUnit = 'min';  %%% internal time unit
    
end;

%%% determine dosing events
%%%
for id=listOfIds.estimation
    
    dosingEvent = GenericPBPKmodel_determineDosingEvents(individual(id));
    individual(id).pred.dosingEvent = dosingEvent;
    
end;


%%% ======================================================================
%%% begin parameters to be estimated
%%%
%%% parameters to be estimated


CLpla_tot=1e-7 ;

n = 1; param(n) = CLpla_tot  ;  LowerBound(n) =0; UpperBound(n) =1;

options=optimset('Display','iter','PlotFcns',@optimplotfval, 'OutputFcn',@report_on_interation);

[P,fval,exitflag,output] = fminsearchbnd(@determine_residuals,param,LowerBound,UpperBound,options,individual,listOfIds);

estimation.listOfIds = listOfIds;
estimation.P         = P;
estimation.fval      = fval;
estimation.exitflg   = exitflag;
estimation.output    = output;


%%% calculate loglikelihood
[res,n]= determine_residuals(P,individual,listOfIds);
sigma2 = res/n;
logL   =-1/2.*(n.*(log(2.*pi.*sigma2))+(1./sigma2).*(res));
%%%
%%% end parameters to be estimated
%%% ======================================================================

%%% assign estimated values to individuals
%%%
for id=listOfIds.estimation
    
    individual(id).estimation.CLpla_tot   = P(1);
    
end;

%%% report about results
%%%
fprintf('\n   Results of the estimation process:\n');
fprintf('\n   (a) estimated parameters: '); fprintf('%2.6e ',estimation.P);
fprintf('\n   (b) optimized residuals: %2.6e',estimation.fval);
fprintf('\n   (c) message: %s',estimation.output.message);
fprintf('\n   (d) loglikelihood: %2.6e',logL);
fprintf('\n   (e) -2*loglikelihood: %2.6e',(-2*logL));
fprintf('\n\n')


printResults=0;
if printResults;
    
    
    %%save results in text file (RunSummary) each for  WT mice, SCID mice
    
    
    if isfield(individual(id).model,'WTmice') && strcmp(individual(id).model.WTmice,'yes')
        
        Run=1;
        
        fid=fopen('EstimationOutputWTmice_linearCL_likelihoodProfiling.txt','at');
        fprintf(fid, '\n   Date %s\n', datestr(now));
        fprintf(fid, '\n   Results of the estimation process: Run %d\n', Run);
        fprintf(fid, '\n   (a) estimated parameters: %2.6e ',estimation.P );
        fprintf(fid, '\n   (b) optimized residuals (not normalized by datapoints): %2.6e', estimation.fval);
        fprintf(fid, '\n   (c) message: %s',estimation.output.message);
        fprintf(fid, '\n   (d) loglikelihood: %2.6e',logL);
        fprintf(fid, '\n   (e) -2*loglikelihood: %2.6e',(-2*logL));
        fprintf(fid, '\n   (f) comments: CLpla_tot estimated');
        fprintf(fid, '\n   (g) Residuals:sum( ( (C_pred-C_exp) ).^2 ) ');
        fclose(fid);
        
        
    end
    
    
    
    if isfield(individual(id).model,'SCID') && strcmp(individual(id).model.SCID,'yes')
        
        Run=1;
        
        fid=fopen('EstimationOutputSCID_linearCL_likelihoodprofiling.txt','at');
        fprintf(fid, '\n   Date %s\n', datestr(now));
        fprintf(fid, '\n   Results of the estimation process: Run %d\n', Run);
        fprintf(fid, '\n   (a) estimated parameters: %2.6e ',estimation.P );
        fprintf(fid, '\n   (b) optimized residuals (not normalized by datapoints): %2.6e', estimation.fval);
        fprintf(fid, '\n   (c) message: %s',estimation.output.message);
        fprintf(fid, '\n   (d) loglikelihood: %2.6e',logL);
        fprintf(fid, '\n   (e) -2*loglikelihood: %2.6e',(-2*logL));
        fprintf(fid, '\n   (f) comments: CLpla_tot estimated');
        fprintf(fid, '\n   (g) Residuals:sum( ( (C_pred-C_exp) ).^2 ) ');
        fclose(fid);
        
    end
    
    if isfield(individual(id).model,'NudeMice') && strcmp(individual(id).model.NudeMice,'yes')
        
        Run=1;
        
        fid=fopen('EstimationOutputNude_linearCL_likelihoodprofiling.txt','at');
        fprintf(fid, '\n   Date %s\n', datestr(now));
        fprintf(fid, '\n   Results of the estimation process: Run %d\n', Run);
        fprintf(fid, '\n   (a) estimated parameters: %2.6e ',estimation.P );
        fprintf(fid, '\n   (b) optimized residuals (not normalized by datapoints): %2.6e', estimation.fval);
        fprintf(fid, '\n   (c) message: %s',estimation.output.message);
        fprintf(fid, '\n   (d) loglikelihood: %2.6e',logL);
        fprintf(fid, '\n   (e) -2*loglikelihood: %2.6e',(-2*logL));
        fprintf(fid, '\n   (f) comments: CLpla_tot estimated');
        fprintf(fid, '\n   (g) Residuals:sum( ( (C_pred-C_exp) ).^2 ) ');
        fclose(fid);
        
    end
    
end
end
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% END: MAIN FUNCTION
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
%%% BEGIN: LOCAL SUB-ROUTINES

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [individual, listOfIds] = GenericPBPKmodel_checkForEstimationModusOn(individual)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

listOfIds.estimation = [];

N_ids = length(individual);
for id = 1:N_ids
    
    %%% report about characteristics of the individual
    individual(id).id = id;
    
    %%% check, if field 'estimation.modus' exists and is equal to 'on'
    %%%
    if isfield(individual(id),'estimation') && isfield(individual(id).estimation,'modus') && strcmp(individual(id).estimation.modus,'on')
        
        listOfIds.estimation = [listOfIds.estimation id];
        
        %%% check, if expIds really only contains ids which are part of the
        %%% virtual population
        %%%
        if ~all(ismember(individual(id).estimation.expIds,[1:N_ids]))
            message = sprintf('individual(%d).estimation.expIds contains ids that are not part of the virtual population!',id);
            GenericPBPKmodel_ReportErrorMessage(message);
        end;
        
        %%% check, if expIds really only contains ids with experimental data
        %%%
        for expId = individual(id).estimation.expIds
            
            if ~strcmp(individual(expId).model.type,'ExperimentalData')
                message = sprintf('individual(%d).estimation.expIds contains ids that do not contain experimental data!',id);
                GenericPBPKmodel_ReportErrorMessage(message);
            end;
            
        end;
        %%% field 'estimation.modus' does not exist or is equal to 'off'
        %%%
    else
        individual(id).estimation.modus = 'off';
    end;
    
end;

%%% compile list of ids with experimental data
%%%
listOfIds.expData = [];
for id = 1:N_ids
    
    if strcmp(individual(id).model.type,'ExperimentalData')
        listOfIds.expData = [listOfIds.expData id];
    end;
    
end;
end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function [res,n] = determine_residuals(param,individual,listOfIds)

%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

%%% assign parameter to be estimated
%%%
%%% assign estimated values to individuals (keep same order as above !!!)
%%%
for id=listOfIds.estimation
    
    individual(id).estimation.CLpla_tot   =param(1);
end;

%%% simulate individual models
%%%
for id=listOfIds.estimation
    
    individual(id) = eval(sprintf('GenericPBPKmodel_%s(individual(id))',individual(id).model.type));
    
end;


%%% compare predictions to experimental data based on standard output,
%%% i.e., vascular plasma and tissue concentration
%%%

%%%log transform data and predictions when using no weighing scheme for SSE
%%%
logtransformation=1;

res = 0; n = 0;
for id=listOfIds.estimation
    
    S = individual(id).model.S;
    pla = S.pla;
    
    if logtransformation==1;
        t_pred = log(individual(id).stdout.t);
    else
        t_pred = individual(id).stdout.t;
    end
    
    for k=individual(id).estimation.expIds;
        
        if logtransformation==1;
            t_exp = log(individual(k).stdout.t) ;
        else
            t_exp = individual(k).stdout.t;
        end;
        [~,ind_pred,ind_exp] = intersect(t_pred,t_exp);
        
        %%% compute residuals for plasma
        %%%
        if logtransformation==1;
            C_pred  = log(individual(id).stdout.C.pla(ind_pred));
            C_exp   = log(individual( k).stdout.C.pla(ind_exp));
            [ind,~] = find(~isnan(C_exp)); % only none NaN values
        else
            C_pred  = individual(id).stdout.C.pla(ind_pred);
            C_exp   = individual( k).stdout.C.pla(ind_exp);
            [ind,~] = find(~isnan(C_exp)); % only none NaN values
        end;
        
        res     = res + compute_residuals(C_pred(ind),C_exp(ind));
        n       = n + length(ind);
        
        
    end;
end;


end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function res = compute_residuals(C_pred,C_exp)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

logtransformation=1;


if logtransformation==1;
    res = sum( ( (C_pred-C_exp) ).^2 );
else
    res = sum( ( (C_pred-C_exp)./C_pred ).^2 );
    
end

end


%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
function stop = report_on_interation(P, optimvalues, state)
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

stop = false;
hold on;
fprintf(' %3.2e ',P);

end








%%% END: LOCAL SUB-ROUTINES
%%% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

